package net.damschen.swatchit.ui.viewmodels

import androidx.lifecycle.ViewModel
import androidx.lifecycle.viewModelScope
import dagger.assisted.Assisted
import dagger.assisted.AssistedFactory
import dagger.assisted.AssistedInject
import dagger.hilt.android.lifecycle.HiltViewModel
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.launch
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.domain.resultWrappers.CalculationResult
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult
import net.damschen.swatchit.ui.models.GaugeCalculationState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.MeasurementFormState
import net.damschen.swatchit.ui.models.MeasurementsState
import net.damschen.swatchit.ui.models.SwatchState
import net.damschen.swatchit.ui.models.ValidatedInput
import net.damschen.swatchit.ui.models.toValidatedInput

@HiltViewModel(assistedFactory = MeasurementViewModel.Factory::class)
class MeasurementViewModel @AssistedInject constructor(
    @Assisted private val swatchId: Int, private val swatchRepository: SwatchRepository
) : ViewModel() {
    private val _loadState = MutableStateFlow<LoadState>(LoadState.Initial)
    val loadState: StateFlow<LoadState> = _loadState.asStateFlow()

    private val _gaugeCalculationState =
        MutableStateFlow<GaugeCalculationState>(GaugeCalculationState.NotCalculated)
    val gaugeCalculationState: StateFlow<GaugeCalculationState> =
        _gaugeCalculationState.asStateFlow()

    private val _savedSuccessfully = MutableSharedFlow<Boolean>()
    val savedSuccessfully: SharedFlow<Boolean> = _savedSuccessfully.asSharedFlow()

    private val _deletedSuccessfully = MutableSharedFlow<Boolean>()
    val deletedSuccessfully: SharedFlow<Boolean> = _deletedSuccessfully.asSharedFlow()

    private val _gaugeSuccessfullySaved = MutableSharedFlow<Boolean>()
    val gaugeSuccessfullySaved: SharedFlow<Boolean> = _gaugeSuccessfullySaved.asSharedFlow()

    private val _measurementFormState = MutableStateFlow<MeasurementFormState>(
        MeasurementFormState.Stitches(
            ValidatedInput.Valid(), ValidatedInput.Valid()
        )
    )
    val measurementFormState: StateFlow<MeasurementFormState> = _measurementFormState

    private var _measurementsState = MutableStateFlow(MeasurementsState(emptyList()))
    val measurementsState: StateFlow<MeasurementsState> = _measurementsState

    private var _swatchState = MutableStateFlow<SwatchState?>(null)

    init {
        initFormState()
        loadSwatch()
    }

    @AssistedFactory
    interface Factory {
        fun create(swatchId: Int): MeasurementViewModel
    }

    fun loadSwatch() {
        viewModelScope.launch {
            _loadState.value = LoadState.Loading
            when (val result = swatchRepository.get(swatchId)) {
                is DatabaseResult.Success -> {
                    result.data?.let { swatch ->
                        _swatchState.value = SwatchState.from(swatch)
                        _measurementsState.value =
                            MeasurementsState.fromMeasurements(swatch.measurements)
                        _loadState.value = LoadState.Success
                    } ?: run {
                        _loadState.value = LoadState.NotFound
                    }
                }

                is DatabaseResult.Error -> {
                    _loadState.value = LoadState.Error(result.exception.message)
                }
            }
        }
    }

    fun addMeasurement() {
        viewModelScope.launch {
            val swatchState = _swatchState.value
            if (!_measurementFormState.value.isValid() || swatchState == null)
                return@launch _savedSuccessfully.emit(false)

            val swatch =
                swatchState.toSwatch()
                    .withMeasurement(_measurementFormState.value.toMeasurement())

            val updateResult = swatchRepository.update(swatch)
            val isSuccess = updateResult is DatabaseResult.Success

            if (isSuccess) {
                _swatchState.update { SwatchState.from(swatch) }
                _measurementsState.value =
                    MeasurementsState.fromMeasurements(swatch.measurements)
            }
            _savedSuccessfully.emit(isSuccess)
        }
    }

    fun deleteMeasurementAt(index: Int) {
        viewModelScope.launch {
            val currentSwatch = _swatchState.value ?: return@launch _savedSuccessfully.emit(false)

            var swatch: Swatch

            try {
                swatch = currentSwatch.toSwatch().withoutMeasurementAt(index)
            } catch (_: IllegalArgumentException) {
                return@launch _deletedSuccessfully.emit(false)
            }
            val updateResult = swatchRepository.update(swatch)
            val isSuccess = updateResult is DatabaseResult.Success
            if (isSuccess) {
                _swatchState.update { SwatchState.from(swatch) }
                _measurementsState.value =
                    MeasurementsState.fromMeasurements(swatch.measurements)
            }

            _deletedSuccessfully.emit(isSuccess)
        }
    }

    fun initFormState() {
        onCountChanged("")
        onSizeChanged("")
        onTypeChanged(true)
    }

    fun onCountChanged(value: String) {
        _measurementFormState.update { current ->
            val newCount = value.toValidatedInput(GaugeCount.validate(value))
            when (current) {
                is MeasurementFormState.Stitches -> current.copy(count = newCount)
                is MeasurementFormState.Rows -> current.copy(count = newCount)
            }
        }
    }

    fun onSizeChanged(value: String) {
        _measurementFormState.update { current ->
            val newSize = value.toValidatedInput(GaugeSize.validate(value))
            when (current) {
                is MeasurementFormState.Stitches -> current.copy(size = newSize)
                is MeasurementFormState.Rows -> current.copy(size = newSize)
            }
        }
    }

    fun onTypeChanged(toStitches: Boolean) {
        _measurementFormState.update { current ->
            if (toStitches) {
                MeasurementFormState.Stitches(current.count, current.size)
            } else {
                MeasurementFormState.Rows(current.count, current.size)
            }
        }
    }

    fun calculateGauge() {
        val swatchState = _swatchState.value ?: run {
            _gaugeCalculationState.value = GaugeCalculationState.NotCalculated
            return
        }

        val result = swatchState.toSwatch().calculateGaugeFromMeasurements()

        if (result is CalculationResult.Error)
            _gaugeCalculationState.value = GaugeCalculationState.Invalid(result.errorMessageId)
        else if (result is CalculationResult.Success)
            _gaugeCalculationState.value = GaugeCalculationState.fromGauge(result.data)
    }

    fun saveGaugeCalculationState() {
        fun gaugeStateToSwatch(): Swatch? {
            val oldSwatchState = _swatchState.value ?: return null

            val gaugeState = when (val calcState = _gaugeCalculationState.value) {
                is GaugeCalculationState.Valid -> calcState.gaugeState
                is GaugeCalculationState.Invalid,
                is GaugeCalculationState.NotCalculated -> return null
            }

            val gauge = gaugeState.toGauge()
            return oldSwatchState.toSwatch().withUpdatedGauge(gauge)
        }

        viewModelScope.launch {
            val swatch = gaugeStateToSwatch()
            if (swatch == null) {
                _gaugeSuccessfullySaved.emit(false)
                return@launch
            }

            val result = swatchRepository.update(swatch)
            val success = result is DatabaseResult.Success

            if (success) {
                _swatchState.update { SwatchState.from(swatch) }
            }

            _gaugeSuccessfullySaved.emit(success)
        }
    }
}