package net.damschen.swatchit.ui.screens

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.heightIn
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.layout.widthIn
import androidx.compose.foundation.lazy.LazyColumn
import androidx.compose.foundation.lazy.items
import androidx.compose.foundation.shape.RoundedCornerShape
import androidx.compose.foundation.text.TextAutoSize
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.filled.Add
import androidx.compose.material3.CardDefaults
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.ElevatedCard
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FabPosition
import androidx.compose.material3.FloatingActionButton
import androidx.compose.material3.Icon
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.clip
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.layout.ContentScale
import androidx.compose.ui.platform.LocalContext
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import androidx.hilt.lifecycle.viewmodel.compose.hiltViewModel
import coil3.compose.AsyncImage
import net.damschen.swatchit.R
import net.damschen.swatchit.domain.extensions.toPlainStringWithoutTrailingZeros
import net.damschen.swatchit.infrastructure.services.PhotoStorageService
import net.damschen.swatchit.ui.elements.ErrorMessage
import net.damschen.swatchit.ui.elements.ProgressIndicator
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.models.SwatchListItem
import net.damschen.swatchit.ui.viewmodels.SwatchListViewModel
import java.io.File

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun SwatchListScreen(
    viewModel: SwatchListViewModel = hiltViewModel(),
    onAddSwatchClicked: () -> Unit,
    onSwatchDetailsClicked: (Int) -> Unit
) {
    val swatches by viewModel.swatches.collectAsState()
    val loadState by viewModel.loadState.collectAsState()
    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())

    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            CenterAlignedTopAppBar(
                colors = TopAppBarDefaults.topAppBarColors(
                    containerColor = MaterialTheme.colorScheme.primaryContainer,
                    titleContentColor = MaterialTheme.colorScheme.primary
                ),
                title = {
                    Text(
                        stringResource(R.string.app_name),
                        maxLines = 1,
                        overflow = TextOverflow.Ellipsis
                    )
                },
                scrollBehavior = scrollBehavior,
            )
        },
        floatingActionButton = {
            FloatingActionButton(
                onClick = onAddSwatchClicked, modifier = Modifier.testTag("AddButton")
            ) {
                Icon(Icons.Filled.Add, stringResource(R.string.add_swatch_description))
            }
        }, floatingActionButtonPosition = FabPosition.Center
    ) { innerPadding ->

        when (val state = loadState) {
            LoadState.Loading -> {
                ProgressIndicator(innerPadding)
            }

            is LoadState.Error -> {
                ErrorMessage(
                    message = state.message ?: stringResource(R.string.unknown_error),
                    onRetry = { viewModel.loadSwatches() }, padding = innerPadding
                )
            }

            LoadState.NotFound -> {}

            LoadState.Success ->

                SwatchesColumn(innerPadding, swatches, onSwatchDetailsClicked)

            LoadState.Initial -> {}
        }
    }
}

@Composable
private fun SwatchesColumn(
    innerPadding: PaddingValues,
    swatches: List<SwatchListItem>,
    onSwatchDetailsClicked: (Int) -> Unit
) {
    Column {
        LazyColumn(
            modifier = Modifier
                .padding(innerPadding)
                .testTag("Column"),
            contentPadding = PaddingValues(horizontal = 16.dp, vertical = 16.dp),
            verticalArrangement = Arrangement.spacedBy(16.dp)
        ) {
            items(swatches, key = { it.id }) { swatch ->
                SwatchItem(swatch, onSwatchDetailsClicked)
            }
        }
    }
}

@Composable
private fun SwatchItem(swatch: SwatchListItem, onSwatchDetailsClicked: (Int) -> Unit) {
    Column {
        ElevatedCard(
            elevation = CardDefaults.cardElevation(
                defaultElevation = 6.dp
            ), modifier = Modifier
                .fillMaxWidth()
                .padding(4.dp), onClick = { onSwatchDetailsClicked(swatch.id) }
        ) {
            Text(
                text = swatch.name,
                modifier = Modifier
                    .padding(16.dp, 16.dp, 8.dp, 0.dp)
                    .align(Alignment.CenterHorizontally),
                style = MaterialTheme.typography.headlineMedium,
                softWrap = false,
                autoSize = TextAutoSize.StepBased(12.sp, 30.sp, 1.sp),
                overflow = TextOverflow.Ellipsis
            )

            val context = LocalContext.current
            val file = swatch.photoFileName?.let {
                File(context.filesDir, "${PhotoStorageService.IMAGES_DIR_NAME}/$it")
            }
            file?.let {
                Box(
                    modifier = Modifier
                        .fillMaxWidth()
                        .padding(48.dp, 12.dp),
                    contentAlignment = Alignment.Center
                ) {
                    AsyncImage(
                        model = file,
                        contentDescription = stringResource(R.string.photo_of_the_swatch),
                        modifier = Modifier
                            .clip(RoundedCornerShape(12.dp))
                            .testTag("Photo_${swatch.id}")
                            .widthIn(max = 300.dp)
                            .heightIn(max = 200.dp),
                        contentScale = ContentScale.Fit
                    )
                }
            }
            Row {
                Text(
                    text = swatch.yarnName,
                    modifier = Modifier
                        .padding(16.dp, 0.dp, 4.dp, 4.dp)
                        .weight(2f, true),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )

                Text(
                    text = swatch.gaugeState?.let {
                        stringResource(
                            R.string.sts_x_rows_cm,
                            it.nrOfStitches,
                            it.nrOfRows,
                            it.gaugeSize.toPlainStringWithoutTrailingZeros()
                        )
                    } ?: stringResource(R.string.no_gauge),
                    modifier = Modifier
                        .padding(8.dp, 8.dp, 16.dp, 0.dp)
                        .weight(2f, true),
                    style = MaterialTheme.typography.titleSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 20.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )
            }
            Row {
                Text(
                    text = swatch.yarnManufacturer,
                    modifier = Modifier
                        .padding(16.dp, 0.dp, 4.dp, 12.dp)
                        .weight(2f, true),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )
                Text(
                    text = stringResource(R.string.x_needles, swatch.needleSize.displayValue),
                    modifier = Modifier
                        .padding(8.dp, 4.dp, 16.dp, 16.dp)
                        .weight(2f, true),
                    style = MaterialTheme.typography.bodyMedium,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 20.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )
            }
        }
    }
}