package net.damschen.swatchit.ui.screens

import net.damschen.swatchit.ui.models.ValidatedInput
import androidx.activity.compose.BackHandler
import androidx.compose.foundation.layout.Column
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.fillMaxWidth
import androidx.compose.foundation.layout.imePadding
import androidx.compose.foundation.layout.padding
import androidx.compose.foundation.rememberScrollState
import androidx.compose.foundation.text.KeyboardOptions
import androidx.compose.foundation.text.TextAutoSize
import androidx.compose.foundation.verticalScroll
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material.icons.filled.Clear
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.FabPosition
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.MaterialTheme
import androidx.compose.material3.OutlinedTextField
import androidx.compose.material3.Scaffold
import androidx.compose.material3.SnackbarHost
import androidx.compose.material3.SnackbarHostState
import androidx.compose.material3.Text
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.remember
import androidx.compose.ui.Alignment.Companion.CenterHorizontally
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.input.KeyboardType
import androidx.compose.ui.text.style.TextOverflow
import androidx.compose.ui.unit.dp
import androidx.compose.ui.unit.sp
import net.damschen.swatchit.R
import net.damschen.swatchit.ui.elements.CardCaption
import net.damschen.swatchit.ui.elements.DetailsCard
import net.damschen.swatchit.ui.elements.ErrorMessage
import net.damschen.swatchit.ui.elements.ProgressIndicator
import net.damschen.swatchit.ui.models.CalculationsState
import net.damschen.swatchit.ui.models.LoadState
import net.damschen.swatchit.ui.viewmodels.EditSwatchViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun GaugeCalculationsScreen(
    viewModel: EditSwatchViewModel, onNavigateBack: () -> Unit
) {
    val loadState by viewModel.loadState.collectAsState()
    val snackbarHostState = remember { SnackbarHostState() }
    val calculationsState by viewModel.calculationsState.collectAsState()

    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())

    BackHandler { onNavigateBack() }

    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        snackbarHost = {
            SnackbarHost(hostState = snackbarHostState)
        },
        topBar = {
            CenterAlignedTopAppBar(
                title = {
                    Text(
                        stringResource(R.string.calculations),
                        maxLines = 1,
                        overflow = TextOverflow.Ellipsis,
                        modifier = Modifier.testTag("CalculationsTitle")
                    )
                }, navigationIcon = {
                    IconButton(
                        onClick = onNavigateBack, modifier = Modifier.testTag("BackButton")
                    ) {
                        Icon(
                            imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                            contentDescription = stringResource(R.string.back_button_description)
                        )
                    }
                }, scrollBehavior = scrollBehavior
            )
        },
        floatingActionButtonPosition = FabPosition.Center
    ) { innerPadding ->
        when (val state = loadState) {
            LoadState.Loading -> {
                ProgressIndicator(innerPadding)
            }

            is LoadState.Error -> {
                ErrorMessage(
                    message = state.message ?: stringResource(R.string.unknown_error),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding,
                    testTag = "LoadStateError"
                )
            }

            LoadState.NotFound -> {
                ErrorMessage(
                    message = stringResource(R.string.swatch_not_found),
                    onRetry = { viewModel.loadSwatch() },
                    padding = innerPadding,
                    testTag = "LoadStateNotFound"
                )
            }

            LoadState.Success -> {
                CalculationsColumn(
                    innerPadding = innerPadding,
                    calculationsState = calculationsState, viewModel = viewModel
                )
            }

            LoadState.Initial -> {}
        }
    }
}

@Composable
fun CalculationsColumn(
    innerPadding: PaddingValues,
    calculationsState: CalculationsState,
    viewModel: EditSwatchViewModel
) {
    Column(
        modifier = Modifier
            .padding(innerPadding)
            .verticalScroll(rememberScrollState())
            .imePadding() // to allow for scrolling when keyboard is up
    ) {
        DetailsCard {
            CardCaption(
                text = stringResource(R.string.horizontal)
            )
            Column(horizontalAlignment = CenterHorizontally) {
                NumericTextField(
                    validatedInput = calculationsState.width,
                    labelText = stringResource(id = R.string.width_cm),
                    modifier = Modifier.testTag("CalculationWidth"),
                    onValueChange = { viewModel.onCalculationWidthChange(it) }
                )

                Text(
                    text = "\u2259",
                    modifier = Modifier
                        .align(CenterHorizontally)
                        .padding(24.dp, 0.dp),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )

                NumericTextField(
                    validatedInput = calculationsState.nrOfStitches,
                    labelText = stringResource(id = R.string.nr_of_stitches),
                    modifier = Modifier.testTag("CalculationStitches"),
                    onValueChange = { viewModel.onCalculationStitchesChange(it) }
                )
            }
        }

        DetailsCard {
            CardCaption(
                text = stringResource(R.string.vertical)
            )
            Column(horizontalAlignment = CenterHorizontally) {
                NumericTextField(
                    validatedInput = calculationsState.height,
                    labelText = stringResource(id = R.string.height_cm),
                    modifier = Modifier.testTag("CalculationHeight"),
                    onValueChange = { viewModel.onCalculationHeightChange(it) }
                )

                Text(
                    text = "\u2259",
                    modifier = Modifier
                        .align(CenterHorizontally)
                        .padding(24.dp, 0.dp),
                    style = MaterialTheme.typography.headlineSmall,
                    softWrap = false,
                    autoSize = TextAutoSize.StepBased(12.sp, 24.sp, 1.sp),
                    overflow = TextOverflow.Ellipsis
                )

                NumericTextField(
                    validatedInput = calculationsState.nrOfRows,
                    labelText = stringResource(id = R.string.nr_of_rows),
                    modifier = Modifier.testTag("CalculationRows"),
                    onValueChange = { viewModel.onCalculationRowsChange(it) }
                )
            }
        }
    }
}

@Composable
fun NumericTextField(
    validatedInput: ValidatedInput,
    onValueChange: (String) -> Unit,
    labelText: String,
    modifier: Modifier = Modifier,
) {

    OutlinedTextField(
        value = validatedInput.value,
        modifier = modifier
            .padding(24.dp, 6.dp)
            .fillMaxWidth(),
        onValueChange = onValueChange,
        label = { Text(text = labelText) },
        isError = validatedInput is ValidatedInput.Invalid,
        keyboardOptions = KeyboardOptions(keyboardType = KeyboardType.Number),
        singleLine = true,
        trailingIcon = {
            IconButton(onClick = { onValueChange("") }, content = {
                Icon(
                    imageVector = Icons.Default.Clear,
                    contentDescription = stringResource(R.string.clear_input)
                )
            }
            )
        },
        supportingText = {
            if (validatedInput is ValidatedInput.Invalid) {
                Text(
                    text = stringResource(validatedInput.errorMessageId),
                    style = MaterialTheme.typography.bodySmall,
                    color = MaterialTheme.colorScheme.onErrorContainer
                )
            }
        },
    )
}


