package net.damschen.swatchit.ui.screens

import androidx.activity.compose.BackHandler
import androidx.compose.material.icons.Icons
import androidx.compose.material.icons.automirrored.filled.ArrowBack
import androidx.compose.material3.CenterAlignedTopAppBar
import androidx.compose.material3.ExperimentalMaterial3Api
import androidx.compose.material3.Icon
import androidx.compose.material3.IconButton
import androidx.compose.material3.Scaffold
import androidx.compose.material3.Text
import androidx.compose.material3.TextButton
import androidx.compose.material3.TopAppBarDefaults
import androidx.compose.material3.rememberTopAppBarState
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.nestedscroll.nestedScroll
import androidx.compose.ui.platform.testTag
import androidx.compose.ui.res.stringResource
import androidx.compose.ui.text.style.TextOverflow
import androidx.hilt.lifecycle.viewmodel.compose.hiltViewModel
import net.damschen.swatchit.R
import net.damschen.swatchit.ui.elements.ErrorDialog
import net.damschen.swatchit.ui.elements.SwatchPropertiesColumn
import net.damschen.swatchit.ui.viewmodels.AddSwatchViewModel

@OptIn(ExperimentalMaterial3Api::class)
@Composable
fun AddSwatchScreen(viewModel: AddSwatchViewModel = hiltViewModel(), onNavigateBack: () -> Unit) {
    val swatchState by viewModel.formManager.swatchFormState.collectAsState()
    val scrollBehavior = TopAppBarDefaults.pinnedScrollBehavior(rememberTopAppBarState())
    var showSaveErrorDialog by remember { mutableStateOf(false) }

    LaunchedEffect(Unit) {
        viewModel.savedSuccessfully.collect { success ->
            if (success)
                onNavigateBack()
            else
                showSaveErrorDialog = true
        }
    }

    BackHandler { onNavigateBack() }
    Scaffold(
        modifier = Modifier.nestedScroll(scrollBehavior.nestedScrollConnection),
        topBar = {
            CenterAlignedTopAppBar(
                title = {
                    Text(
                        stringResource(R.string.add_swatch),
                        maxLines = 1,
                        overflow = TextOverflow.Ellipsis
                    )
                }, navigationIcon = {
                    IconButton(onClick = {
                        onNavigateBack()
                    }, modifier = Modifier.testTag("CancelButton")) {
                        Icon(
                            imageVector = Icons.AutoMirrored.Filled.ArrowBack,
                            contentDescription = stringResource(R.string.back_button_description)
                        )
                    }
                }, actions = {
                    TextButton(modifier = Modifier.testTag("AddSwatchButton"), onClick = {
                        viewModel.addSwatch()
                    }, enabled = swatchState.isValid()) { Text(stringResource(R.string.save)) }
                }, scrollBehavior = scrollBehavior
            )
        }) { innerPadding ->
        SwatchPropertiesColumn(innerPadding, viewModel.formManager, swatchState)

        if (showSaveErrorDialog) {
            ErrorDialog(
                text = stringResource(R.string.error_while_saving_swatch),
                onRetry = {
                    showSaveErrorDialog = false
                    viewModel.addSwatch()
                },
                onCancel = {
                    showSaveErrorDialog = false
                },
                testTag = "AddErrorDialog"
            )
        }
    }
}


