package net.damschen.swatchit.ui.models

import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.update
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchName
import net.damschen.swatchit.domain.aggregates.swatch.YarnManufacturer
import net.damschen.swatchit.domain.aggregates.swatch.YarnName
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import net.damschen.swatchit.ui.providers.DateTimeProvider

class SwatchFormStateManager(dateTimeProvider: DateTimeProvider) {
    private val defaultCreatedAt: Long = dateTimeProvider.now().time
    private val _swatchFormState = MutableStateFlow(
        SwatchFormState.init(defaultCreatedAt)
    )
    val swatchFormState: StateFlow<SwatchFormState> = _swatchFormState

    val swatchFormIsValid get() = _swatchFormState.value.isValid()
    fun resetFormState() {
        _swatchFormState.value = SwatchFormState.init(defaultCreatedAt)
    }

    fun mergeStateWith(swatch: Swatch) = swatchFormState.value.mergeWith(swatch)

    fun reloadSwatchFormState(swatchState: SwatchState) {
        _swatchFormState.value = SwatchFormState.fromSwatchState(swatchState)
    }

    fun onNameChange(value: String) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                name = value.toValidatedInput(SwatchName.validate(value))
            )
        }
    }

    fun onYarnManufacturerChange(value: String) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                yarnManufacturer = value.toValidatedInput(YarnManufacturer.validate(value))
            )
        }
    }

    fun onYarnNameChange(value: String) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                yarnName = value.toValidatedInput(YarnName.validate(value))
            )
        }
    }

    fun onNeedleSizeChange(value: KnittingNeedleSize) {
        _swatchFormState.update { currentState ->
            currentState.copy(needleSize = value)
        }
    }

    fun onPatternChange(value: String) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                pattern = value.toValidatedInput(Pattern.validate(value))
            )
        }
    }

    fun onNotesChange(value: String) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                notes = value.toValidatedInput(Notes.validate(value))
            )
        }
    }

    fun onCreatedAtChange(value: Long) {
        _swatchFormState.update { currentState ->
            currentState.copy(
                createdAt = value
            )
        }
    }
}