package net.damschen.swatchit.ui.models

import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.SwatchName
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.aggregates.swatch.YarnManufacturer
import net.damschen.swatchit.domain.aggregates.swatch.YarnName
import net.damschen.swatchit.ui.enums.KnittingNeedleSize
import net.damschen.swatchit.ui.enums.toDataEnum
import net.damschen.swatchit.ui.enums.toUiEnum
import java.text.SimpleDateFormat
import java.util.Locale

data class SwatchFormState(
    val yarnManufacturer: ValidatedInput,
    val yarnName: ValidatedInput,
    val pattern: ValidatedInput,
    val needleSize: KnittingNeedleSize,
    val notes: ValidatedInput,
    val createdAt: Long,
    val name: ValidatedInput,
    val id: Int? = null
) {
    companion object {
        fun fromSwatchState(swatch: SwatchState): SwatchFormState {
            requireNotNull(swatch.id)
            return SwatchFormState(
                yarnManufacturer = ValidatedInput.Valid(swatch.yarn?.manufacturer?.value ?: ""),
                yarnName = ValidatedInput.Valid(swatch.yarn?.name?.value ?: ""),
                pattern = ValidatedInput.Valid(swatch.pattern?.value ?: ""),
                needleSize = swatch.needleSize.toUiEnum(),
                notes = ValidatedInput.Valid(swatch.notes?.value ?: ""),
                createdAt = swatch.createdAt.value,
                id = swatch.id.value,
                name = ValidatedInput.Valid(swatch.name?.value ?: "")
            )
        }

        fun init(createdAt: Long): SwatchFormState {
            val defaultNeedleSize: KnittingNeedleSize = KnittingNeedleSize.SIZE_2_5
            return SwatchFormState(
                yarnManufacturer = ValidatedInput.Valid(),
                yarnName = ValidatedInput.Valid(),
                pattern = ValidatedInput.Valid(),
                needleSize = defaultNeedleSize,
                notes = ValidatedInput.Valid(),
                createdAt = createdAt,
                name = ValidatedInput.Valid(),
                id = null
            )
        }
    }

    fun isValid(): Boolean =
        yarnManufacturer is ValidatedInput.Valid && yarnName is ValidatedInput.Valid &&
                pattern is ValidatedInput.Valid && notes is ValidatedInput.Valid &&
                name is ValidatedInput.Valid

    fun toSwatch(): Swatch {
        return Swatch.create(
            needleSize = needleSize.toDataEnum(),
            pattern = Pattern.create(pattern.value),
            yarn = Yarn.create(
                YarnName.create(yarnName.value), YarnManufacturer.create(yarnManufacturer.value)
            ),
            notes = Notes.create(notes.value),
            createdAt = EpochMillis(createdAt),
            id = id?.let { SwatchId(id) },
            name = SwatchName.create(name.value)
        )
    }

    fun mergeWith(source: Swatch): Swatch {
        return toSwatch().withNewMeasurements(source.measurements).let { swatch ->
            val merged = source.gauge?.let { swatch.withUpdatedGauge(it) } ?: swatch
            source.photo?.let { merged.withUpdatedPhoto(it) } ?: merged
        }
    }
}

fun Long.toFormattedDate(): String {
    val formatter = SimpleDateFormat("dd/MM/yyyy", Locale.getDefault())
    return formatter.format(this)
}