package net.damschen.swatchit.infrastructure.services

import android.content.Context
import android.graphics.Bitmap
import android.net.Uri
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.infrastructure.resultWrappers.PhotoResult
import java.io.File
import kotlin.coroutines.cancellation.CancellationException

interface PhotoStorageService {
    companion object {
        const val IMAGES_DIR_NAME = "images"
    }

    suspend fun copyToLocalFileSystem(uri: Uri, photo: Photo): PhotoResult
    suspend fun deleteFromLocalFileSystem(photo: Photo): PhotoResult
}

class AndroidPhotoStorageService(
    context: Context,
    private val bitmapDecoder: BitmapDecoder,
    private val ioDispatcher: CoroutineDispatcher
) : PhotoStorageService {

    private val imagesDir = File(context.filesDir, PhotoStorageService.IMAGES_DIR_NAME)

    private companion object {
        const val MAX_IMAGE_SIZE = 1440
        const val JPEG_QUALITY = 85
    }

    override suspend fun copyToLocalFileSystem(uri: Uri, photo: Photo): PhotoResult =
        withContext(ioDispatcher) {
            try {
                ensureDirectoryExists()

                val destinationFile = File(imagesDir, photo.fileName)

                val bitmap = bitmapDecoder.decode(uri, MAX_IMAGE_SIZE)
                destinationFile.outputStream().use { output ->
                    bitmap.compress(Bitmap.CompressFormat.JPEG, JPEG_QUALITY, output)
                }
                PhotoResult.Success
            } catch (e: Exception) {
                if (e is CancellationException) throw e
                PhotoResult.Error(e.message)
            }
        }

    override suspend fun deleteFromLocalFileSystem(photo: Photo): PhotoResult =
        withContext(ioDispatcher) {
            try {
                val file = File(imagesDir, photo.fileName)
                if (file.exists() && !file.delete()) {
                    PhotoResult.Error("Could not delete file.")
                } else {
                    PhotoResult.Success
                }
            } catch (e: Exception) {
                if (e is CancellationException) throw e
                PhotoResult.Error(e.message)
            }
        }

    private fun ensureDirectoryExists() {
        if (!imagesDir.exists() && !imagesDir.mkdirs()) {
            throw FileSystemException(
                imagesDir,
                reason = "Failed to create directory"
            )
        }
    }
}