package net.damschen.swatchit.infrastructure.services

import android.content.Context
import androidx.core.net.toUri
import kotlinx.coroutines.CoroutineDispatcher
import kotlinx.coroutines.withContext
import net.damschen.swatchit.infrastructure.database.AppDatabase
import net.damschen.swatchit.infrastructure.resultWrappers.ExportResult
import net.damschen.swatchit.infrastructure.services.DataExportService.Companion.BACKUP_FILE_NAME
import java.io.File
import kotlin.coroutines.cancellation.CancellationException

interface DataExportService {
    suspend fun zipDatabaseAndPictures(): ExportResult

    companion object {
        const val BACKUP_FILE_NAME = "swatchit_backup.zip"
    }
}

class AndroidDataExportService(
    val context: Context,
    val database: AppDatabase,
    val zipper: Zipper,
    val ioDispatcher: CoroutineDispatcher
) :
    DataExportService {

    private val backupDir = File(context.cacheDir, BACKUP_DIR_NAME)

    companion object {
        private const val DATABASE_BACKUP_NAME = "swatchit_database_backup.db"
        private const val BACKUP_DIR_NAME = "bkp"
        private const val IMAGES_BACKUP_DIR = "images_backup"
    }

    override suspend fun zipDatabaseAndPictures(): ExportResult = withContext(ioDispatcher) {
        val zipFile = File(context.cacheDir, BACKUP_FILE_NAME)
        try {
            if (zipFile.exists()) zipFile.delete()
            ensureDirectoryExists()
            copyDatabase()
            copyPhotoFolder()
            zipper.zipFolder(backupDir, zipFile)

            ExportResult.Success(zipFile.toUri())
        } catch (e: Exception) {
            if (e is CancellationException) throw e
            ExportResult.Error(e.message)
        } finally {
            backupDir.deleteRecursively()
        }
    }

    private fun copyPhotoFolder() {
        val imagesDir = File(context.filesDir, PhotoStorageService.IMAGES_DIR_NAME)
        val imagesBackupDir = File(backupDir, IMAGES_BACKUP_DIR)
        if (imagesDir.exists()) {
            imagesDir.copyRecursively(imagesBackupDir, true)
        }
    }

    private fun copyDatabase() {
        val bkpFile = File(backupDir, DATABASE_BACKUP_NAME)
        if (bkpFile.exists())
            bkpFile.delete()

        database.openHelper.writableDatabase.execSQL("VACUUM INTO '${bkpFile.absolutePath}'")
    }

    private fun ensureDirectoryExists() {
        if (backupDir.exists()) backupDir.deleteRecursively()
        if (!backupDir.mkdirs()) {
            throw FileSystemException(
                backupDir,
                reason = "Failed to create directory"
            )
        }
    }
}