package net.damschen.swatchit.infrastructure.repository

import android.content.res.Resources.NotFoundException
import android.database.sqlite.SQLiteException
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.repositories.SwatchRepository
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult
import net.damschen.swatchit.domain.resultWrappers.DatabaseResult.Success
import net.damschen.swatchit.infrastructure.database.MeasurementDao
import net.damschen.swatchit.infrastructure.database.SwatchDao
import net.damschen.swatchit.infrastructure.database.TransactionProvider
import java.security.InvalidParameterException

class SqlSwatchRepository(
    private val swatchDao: SwatchDao,
    private val measurementDao: MeasurementDao,
    private val transactionProvider: TransactionProvider
) : SwatchRepository {

    private suspend fun <T> safeDbCall(
        dbCall: suspend () -> T
    ): DatabaseResult<T> = try {
        val result = transactionProvider.executeTransaction { dbCall() }
        Success(result)
    } catch (e: SQLiteException) {
        DatabaseResult.Error(e)
    } catch (e: NotFoundException) {
        DatabaseResult.Error(e)
    }

    override fun getAll(): Flow<DatabaseResult<List<Swatch>>> = flow {
        swatchDao.get().collect { swatchAggregates ->
            val swatches = swatchAggregates.map { aggregate ->
                aggregate.toSwatch()
            }
            emit(Success(swatches) as DatabaseResult<List<Swatch>>)
        }
    }.catch { e ->
        if (e is SQLiteException) emit(DatabaseResult.Error(e))
    }

    override suspend fun get(id: Int): DatabaseResult<Swatch?> {
        return safeDbCall {
            swatchDao.get(id)?.toSwatch()
        }
    }

    override suspend fun add(swatch: Swatch): DatabaseResult<Unit> {
        if (swatch.id != null)
            return DatabaseResult.Error(InvalidParameterException("The data of the swatch you are trying to insert seems to be corrupted."))
        val swatchEntity = swatch.toEntity()

        return safeDbCall {
            val id = swatchDao.insert(swatchEntity)
            val measurements = swatch.measurements.toEntities(id.toInt())
            measurementDao.insert(measurements)
        }
    }

    override suspend fun update(swatch: Swatch): DatabaseResult<Unit> {
        val swatchEntity = swatch.toEntity()
        return safeDbCall {
            if (swatch.id == null || !swatchDao.exists(swatch.id.value))
                throw NotFoundException("Swatch does not exists, it might have been deleted.")
            swatchDao.update(swatchEntity)
            measurementDao.deleteAllForSwatch(swatch.id.value)
            val measurements = swatch.measurements.toEntities(swatch.id.value)
            measurementDao.insert(measurements)
        }
    }

    override suspend fun delete(id: SwatchId): DatabaseResult<Unit> {
        return safeDbCall {
            swatchDao.delete(net.damschen.swatchit.infrastructure.database.SwatchId(id.value))
        }
    }
}