package net.damschen.swatchit.infrastructure.repository

import net.damschen.swatchit.domain.aggregates.swatch.EpochMillis
import net.damschen.swatchit.domain.aggregates.swatch.Gauge
import net.damschen.swatchit.domain.aggregates.swatch.GaugeCount
import net.damschen.swatchit.domain.aggregates.swatch.GaugeSize
import net.damschen.swatchit.domain.aggregates.swatch.KnittingNeedleSize
import net.damschen.swatchit.domain.aggregates.swatch.Measurement
import net.damschen.swatchit.domain.aggregates.swatch.Notes
import net.damschen.swatchit.domain.aggregates.swatch.Pattern
import net.damschen.swatchit.domain.aggregates.swatch.Photo
import net.damschen.swatchit.domain.aggregates.swatch.Swatch
import net.damschen.swatchit.domain.aggregates.swatch.SwatchId
import net.damschen.swatchit.domain.aggregates.swatch.SwatchName
import net.damschen.swatchit.domain.aggregates.swatch.Yarn
import net.damschen.swatchit.domain.aggregates.swatch.YarnManufacturer
import net.damschen.swatchit.domain.aggregates.swatch.YarnName
import net.damschen.swatchit.infrastructure.database.MeasurementEntity
import net.damschen.swatchit.infrastructure.database.SwatchAggregate
import net.damschen.swatchit.infrastructure.database.SwatchEntity

fun Swatch.toEntity() = SwatchEntity(
    needleSize = needleSize.toDbNeedleSize(),
    yarnName = yarn?.name?.value ?: "",
    yarnManufacturer = yarn?.manufacturer?.value ?: "",
    nrOfStitches = gauge?.nrOfStitches?.value,
    nrOfRows = gauge?.nrOfRows?.value,
    gaugeLength = gauge?.size?.value,
    createdAt = createdAt.value,
    name = name?.value ?: "",
    pattern = pattern?.value ?: "",
    notes = notes?.value ?: "",
    photoUUID = photo?.identifier
).also { entity ->
    id?.let { entity.id = it.value }
}


fun List<Measurement>.toEntities(swatchId: Int) = map { measurement ->
    MeasurementEntity(
        measurement.toDbMeasurementType(),
        measurement.count.value,
        measurement.size.value,
        swatchId
    )
}

private fun SwatchEntity.toSwatch(): Swatch {
    val knittingNeedleSize = needleSize
    val notes = Notes.create(notes)
    val pattern = Pattern.create(pattern)
    val name = SwatchName.create(name)
    val yarnName = YarnName.create(yarnName)
    val yarnManufacturer = YarnManufacturer.create(yarnManufacturer)
    val yarn = Yarn.create(yarnName, yarnManufacturer)
    val createdAt = EpochMillis(createdAt)
    val photo = photoUUID?.let { Photo(it) }
    var swatch = Swatch.create(
        needleSize = knittingNeedleSize.toNeedleSize(),
        pattern = pattern,
        yarn = yarn,
        notes = notes,
        createdAt = createdAt,
        id = SwatchId(id),
        name = name
    )
    swatch = photo?.let { swatch.withUpdatedPhoto(it) } ?: swatch

    val gauge = gaugeLength?.let { length ->
        nrOfStitches?.let { stitches ->
            nrOfRows?.let { rows ->
                Gauge(
                    GaugeCount(stitches),
                    GaugeCount(rows),
                    GaugeSize(length)
                )
            }
        }
    }

    return gauge?.let { swatch.withUpdatedGauge(it) } ?: swatch
}

fun SwatchAggregate.toSwatch(): Swatch {
    return swatch.toSwatch().withNewMeasurements(measurements.map {
        it.toMeasurement()
    })
}

fun MeasurementEntity.toMeasurement(): Measurement {
    val gaugeCount = GaugeCount(count)
    val gaugeSize = GaugeSize(size)

    return when (type) {
        net.damschen.swatchit.infrastructure.database.MeasurementType.Stitches ->
            Measurement.Stitches(gaugeCount, gaugeSize)
        net.damschen.swatchit.infrastructure.database.MeasurementType.Rows ->
            Measurement.Rows(gaugeCount, gaugeSize)
    }
}

private fun Measurement.toDbMeasurementType() = when (this) {
    is Measurement.Stitches -> net.damschen.swatchit.infrastructure.database.MeasurementType.Stitches
    is Measurement.Rows -> net.damschen.swatchit.infrastructure.database.MeasurementType.Rows
}

private fun KnittingNeedleSize.toDbNeedleSize() = when (this) {
    KnittingNeedleSize.SIZE_0_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_0_5
    KnittingNeedleSize.SIZE_0_75 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_0_75
    KnittingNeedleSize.SIZE_1_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_0
    KnittingNeedleSize.SIZE_1_25 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_25
    KnittingNeedleSize.SIZE_1_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_5
    KnittingNeedleSize.SIZE_1_75 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_75
    KnittingNeedleSize.SIZE_2_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_0
    KnittingNeedleSize.SIZE_2_25 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_25
    KnittingNeedleSize.SIZE_2_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_5
    KnittingNeedleSize.SIZE_2_75 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_75
    KnittingNeedleSize.SIZE_3_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_0
    KnittingNeedleSize.SIZE_3_25 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_25
    KnittingNeedleSize.SIZE_3_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_5
    KnittingNeedleSize.SIZE_3_75 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_75
    KnittingNeedleSize.SIZE_4_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_0
    KnittingNeedleSize.SIZE_4_25 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_25
    KnittingNeedleSize.SIZE_4_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_5
    KnittingNeedleSize.SIZE_4_75 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_75
    KnittingNeedleSize.SIZE_5_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_5_0
    KnittingNeedleSize.SIZE_5_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_5_5
    KnittingNeedleSize.SIZE_6_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_6_0
    KnittingNeedleSize.SIZE_6_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_6_5
    KnittingNeedleSize.SIZE_7_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_7_0
    KnittingNeedleSize.SIZE_7_5 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_7_5
    KnittingNeedleSize.SIZE_8_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_8_0
    KnittingNeedleSize.SIZE_9_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_9_0
    KnittingNeedleSize.SIZE_10_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_10_0
    KnittingNeedleSize.SIZE_12_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_12_0
    KnittingNeedleSize.SIZE_15_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_15_0
    KnittingNeedleSize.SIZE_16_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_16_0
    KnittingNeedleSize.SIZE_19_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_19_0
    KnittingNeedleSize.SIZE_20_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_20_0
    KnittingNeedleSize.SIZE_25_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_25_0
    KnittingNeedleSize.SIZE_30_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_30_0
    KnittingNeedleSize.SIZE_35_0 -> net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_35_0
}

private fun net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.toNeedleSize() =
    when (this) {
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_0_5 -> KnittingNeedleSize.SIZE_0_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_0_75 -> KnittingNeedleSize.SIZE_0_75
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_0 -> KnittingNeedleSize.SIZE_1_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_25 -> KnittingNeedleSize.SIZE_1_25
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_5 -> KnittingNeedleSize.SIZE_1_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_1_75 -> KnittingNeedleSize.SIZE_1_75
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_0 -> KnittingNeedleSize.SIZE_2_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_25 -> KnittingNeedleSize.SIZE_2_25
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_5 -> KnittingNeedleSize.SIZE_2_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_2_75 -> KnittingNeedleSize.SIZE_2_75
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_0 -> KnittingNeedleSize.SIZE_3_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_25 -> KnittingNeedleSize.SIZE_3_25
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_5 -> KnittingNeedleSize.SIZE_3_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_3_75 -> KnittingNeedleSize.SIZE_3_75
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_0 -> KnittingNeedleSize.SIZE_4_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_25 -> KnittingNeedleSize.SIZE_4_25
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_5 -> KnittingNeedleSize.SIZE_4_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_4_75 -> KnittingNeedleSize.SIZE_4_75
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_5_0 -> KnittingNeedleSize.SIZE_5_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_5_5 -> KnittingNeedleSize.SIZE_5_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_6_0 -> KnittingNeedleSize.SIZE_6_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_6_5 -> KnittingNeedleSize.SIZE_6_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_7_0 -> KnittingNeedleSize.SIZE_7_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_7_5 -> KnittingNeedleSize.SIZE_7_5
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_8_0 -> KnittingNeedleSize.SIZE_8_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_9_0 -> KnittingNeedleSize.SIZE_9_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_10_0 -> KnittingNeedleSize.SIZE_10_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_12_0 -> KnittingNeedleSize.SIZE_12_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_15_0 -> KnittingNeedleSize.SIZE_15_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_16_0 -> KnittingNeedleSize.SIZE_16_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_19_0 -> KnittingNeedleSize.SIZE_19_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_20_0 -> KnittingNeedleSize.SIZE_20_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_25_0 -> KnittingNeedleSize.SIZE_25_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_30_0 -> KnittingNeedleSize.SIZE_30_0
        net.damschen.swatchit.infrastructure.database.KnittingNeedleSize.SIZE_35_0 -> KnittingNeedleSize.SIZE_35_0
    }