"use strict";

import { Buffer } from '@craftzdog/react-native-buffer';
import { createSecretKey, CryptoKey } from './keys';
import { lazyDOMException, hasAnyNotIn } from './Utils';
export async function hmacImportKey(algorithm, format, keyData, extractable, keyUsages) {
  // Validate key usages
  if (hasAnyNotIn(keyUsages, ['sign', 'verify'])) {
    throw lazyDOMException('Unsupported key usage for an HMAC key', 'SyntaxError');
  }
  if (!keyData) {
    throw lazyDOMException('Invalid keyData', 'DataError');
  }
  let keyMaterial;
  switch (format) {
    case 'raw':
      {
        // For raw format, keyData should be BufferLike
        if (typeof keyData === 'string') {
          keyMaterial = Buffer.from(keyData, 'base64');
        } else if (Buffer.isBuffer(keyData)) {
          keyMaterial = keyData;
        } else {
          keyMaterial = Buffer.from(keyData);
        }
        break;
      }
    case 'jwk':
      {
        const jwk = keyData;

        // Validate required JWK properties
        if (typeof jwk !== 'object' || jwk.kty !== 'oct' || !jwk.k) {
          throw lazyDOMException('Invalid JWK format for HMAC key', 'DataError');
        }
        if (algorithm.length === 0) {
          throw lazyDOMException('Zero-length key is not supported', 'DataError');
        }

        // The Web Crypto spec allows for key lengths that are not multiples of 8. We don't.
        if (algorithm.length && algorithm.length % 8) {
          throw lazyDOMException('Unsupported algorithm.length', 'NotSupportedError');
        }

        // Convert base64 to buffer
        keyMaterial = Buffer.from(jwk.k, 'base64');

        // If algorithm.length is specified, validate key length
        if (algorithm.length && keyMaterial.length * 8 !== algorithm.length) {
          throw lazyDOMException('Invalid key length', 'DataError');
        }
        break;
      }
    default:
      throw lazyDOMException(`Unable to import HMAC key with format ${format}`, 'NotSupportedError');
  }

  // Create the key object
  const keyObject = createSecretKey(keyMaterial);

  // Return new CryptoKey
  return new CryptoKey(keyObject, algorithm, keyUsages, extractable);
}
//# sourceMappingURL=mac.js.map