"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.pbkdf2 = pbkdf2;
exports.pbkdf2DeriveBits = pbkdf2DeriveBits;
exports.pbkdf2Sync = pbkdf2Sync;
var _NativeQuickCrypto = require("./NativeQuickCrypto/NativeQuickCrypto");
var _reactNativeBuffer = require("@craftzdog/react-native-buffer");
var _Utils = require("./Utils");
var _util = require("util");
const WRONG_PASS = 'Password must be a string, a Buffer, a typed array or a DataView';
const WRONG_SALT = 'Salt must be a string, a Buffer, a typed array or a DataView';
function sanitizeInput(input, errorMsg) {
  try {
    return (0, _Utils.binaryLikeToArrayBuffer)(input);
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
  } catch (_e) {
    throw new Error(errorMsg);
  }
}
const nativePbkdf2 = _NativeQuickCrypto.NativeQuickCrypto.pbkdf2;
function pbkdf2(password, salt, iterations, keylen, digest, callback) {
  if (callback === undefined || typeof callback !== 'function') {
    throw new Error('No callback provided to pbkdf2');
  }
  const sanitizedPassword = sanitizeInput(password, WRONG_PASS);
  const sanitizedSalt = sanitizeInput(salt, WRONG_SALT);
  const normalizedDigest = (0, _Utils.normalizeHashName)(digest, _Utils.HashContext.Node);
  nativePbkdf2.pbkdf2(sanitizedPassword, sanitizedSalt, iterations, keylen, normalizedDigest).then(res => {
    callback(null, _reactNativeBuffer.Buffer.from(res));
  }, e => {
    callback(e);
  });
}
function pbkdf2Sync(password, salt, iterations, keylen, digest) {
  const sanitizedPassword = sanitizeInput(password, WRONG_PASS);
  const sanitizedSalt = sanitizeInput(salt, WRONG_SALT);
  const algo = digest ? (0, _Utils.normalizeHashName)(digest, _Utils.HashContext.Node) : 'sha1';
  const result = nativePbkdf2.pbkdf2Sync(sanitizedPassword, sanitizedSalt, iterations, keylen, algo);
  return _reactNativeBuffer.Buffer.from(result);
}

// We need this because the typescript  overload signatures in pbkdf2() above do
// not play nice with promisify() below.
const pbkdf2WithDigest = (password, salt, iterations, keylen, digest, callback) => pbkdf2(password, salt, iterations, keylen, digest, callback);
const pbkdf2Promise = (0, _util.promisify)(pbkdf2WithDigest);
async function pbkdf2DeriveBits(algorithm, baseKey, length) {
  const {
    iterations,
    hash,
    salt
  } = algorithm;
  const normalizedHash = (0, _Utils.normalizeHashName)(hash);
  if (!normalizedHash) {
    throw (0, _Utils.lazyDOMException)('hash cannot be blank', 'OperationError');
  }
  if (!iterations || iterations === 0) {
    throw (0, _Utils.lazyDOMException)('iterations cannot be zero', 'OperationError');
  }
  if (!salt) {
    throw (0, _Utils.lazyDOMException)(WRONG_SALT, 'OperationError');
  }
  const raw = baseKey.keyObject.export();
  if (length === 0) throw (0, _Utils.lazyDOMException)('length cannot be zero', 'OperationError');
  if (length === null) throw (0, _Utils.lazyDOMException)('length cannot be null', 'OperationError');
  if (length % 8) {
    throw (0, _Utils.lazyDOMException)('length must be a multiple of 8', 'OperationError');
  }
  const sanitizedPassword = sanitizeInput(raw, WRONG_PASS);
  const sanitizedSalt = sanitizeInput(salt, WRONG_SALT);
  const result = await pbkdf2Promise(sanitizedPassword, sanitizedSalt, iterations, length / 8, normalizedHash);
  if (!result) {
    throw (0, _Utils.lazyDOMException)('received bad result from pbkdf2()', 'OperationError');
  }
  return (0, _Utils.bufferLikeToArrayBuffer)(result);
}
//# sourceMappingURL=pbkdf2.js.map