import { EditorState, Plugin, Transaction, Command } from 'prosemirror-state';
import { DecorationSet } from 'prosemirror-view';
import { Slice } from 'prosemirror-model';

declare class SearchQuery {
    /**
    The search string (or regular expression).
    */
    readonly search: string;
    /**
    Indicates whether the search is case-sensitive.
    */
    readonly caseSensitive: boolean;
    /**
    By default, string search will replace `\n`, `\r`, and `\t` in
    the query with newline, return, and tab characters. When this
    is set to true, that behavior is disabled.
    */
    readonly literal: boolean;
    /**
    When true, the search string is interpreted as a regular
    expression.
    */
    readonly regexp: boolean;
    /**
    The replace text, or the empty string if no replace text has
    been given.
    */
    readonly replace: string;
    /**
    Whether this query is non-empty and, in case of a regular
    expression search, syntactically valid.
    */
    readonly valid: boolean;
    /**
    When true, matches that contain words are ignored when there are
    further word characters around them.
    */
    readonly wholeWord: boolean;
    /**
    An optional filter that causes some results to be ignored.
    */
    readonly filter: ((state: EditorState, result: SearchResult) => boolean) | null;
    /**
    Create a query object.
    */
    constructor(config: {
        /**
        The search string.
        */
        search: string;
        /**
        Controls whether the search should be case-sensitive.
        */
        caseSensitive?: boolean;
        /**
        By default, string search will replace `\n`, `\r`, and `\t` in
        the query with newline, return, and tab characters. When this
        is set to true, that behavior is disabled.
        */
        literal?: boolean;
        /**
        When true, interpret the search string as a regular expression.
        */
        regexp?: boolean;
        /**
        The replace text.
        */
        replace?: string;
        /**
        Enable whole-word matching.
        */
        wholeWord?: boolean;
        /**
        Providing a filter causes results for which the filter returns
        false to be ignored.
        */
        filter?: (state: EditorState, result: SearchResult) => boolean;
    });
    /**
    Compare this query to another query.
    */
    eq(other: SearchQuery): boolean;
    /**
    Find the next occurrence of this query in the given range.
    */
    findNext(state: EditorState, from?: number, to?: number): SearchResult | null;
    /**
    Find the previous occurrence of this query in the given range.
    Note that, if `to` is given, it should be _less_ than `from`.
    */
    findPrev(state: EditorState, from?: number, to?: number): SearchResult | null;
    /**
    Get the ranges that should be replaced for this result. This can
    return multiple ranges when `this.replace` contains
    `$1`/`$&`-style placeholders, in which case the preserved
    content is skipped by the replacements.
    
    Ranges are sorted by position, and `from`/`to` positions all
    refer to positions in `state.doc`. When applying these, you'll
    want to either apply them from back to front, or map these
    positions through your transaction's current mapping.
    */
    getReplacements(state: EditorState, result: SearchResult): {
        from: number;
        to: number;
        insert: Slice;
    }[];
}
/**
A matched instance of a search query. `match` will be non-null
only for regular expression queries.
*/
interface SearchResult {
    from: number;
    to: number;
    match: RegExpMatchArray | null;
    matchStart: number;
}

/**
Returns a plugin that stores a current search query and searched
range, and highlights matches of the query.
*/
declare function search(options?: {
    initialQuery?: SearchQuery;
    initialRange?: {
        from: number;
        to: number;
    };
}): Plugin;
/**
Get the current active search query and searched range. Will
return `undefined` is the search plugin isn't active.
*/
declare function getSearchState(state: EditorState): {
    query: SearchQuery;
    range: {
        from: number;
        to: number;
    } | null;
} | undefined;
/**
Access the decoration set holding the currently highlighted search
matches in the document.
*/
declare function getMatchHighlights(state: EditorState): DecorationSet;
/**
Add metadata to a transaction that updates the active search query
and searched range, when dispatched.
*/
declare function setSearchState(tr: Transaction, query: SearchQuery, range?: {
    from: number;
    to: number;
} | null): Transaction;
/**
Find the next instance of the search query after the current
selection and move the selection to it.
*/
declare const findNext: Command;
/**
Find the next instance of the search query and move the selection
to it. Don't wrap around at the end of document or search range.
*/
declare const findNextNoWrap: Command;
/**
Find the previous instance of the search query and move the
selection to it.
*/
declare const findPrev: Command;
/**
Find the previous instance of the search query and move the
selection to it. Don't wrap at the start of the document or search
range.
*/
declare const findPrevNoWrap: Command;
/**
Replace the currently selected instance of the search query, and
move to the next one. Or select the next match, if none is already
selected.
*/
declare const replaceNext: Command;
/**
Replace the next instance of the search query. Don't wrap around
at the end of the document.
*/
declare const replaceNextNoWrap: Command;
/**
Replace the currently selected instance of the search query, if
any, and keep it selected.
*/
declare const replaceCurrent: Command;
/**
Replace all instances of the search query.
*/
declare const replaceAll: Command;

export { SearchQuery, type SearchResult, findNext, findNextNoWrap, findPrev, findPrevNoWrap, getMatchHighlights, getSearchState, replaceAll, replaceCurrent, replaceNext, replaceNextNoWrap, search, setSearchState };
