"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.stripNonAccents = exports.stripAccents = exports.accentForms = exports.caseForms = exports.expandRange = exports.expandCharacterSet = void 0;
/**
 * Expand a line into a set of characters.
 *
 * Example:
 * - `a-c` -> `<a,b,c>`
 * - `ac-` -> `<a,c,->`
 * - `-abz` -> `<-,a,b,z>`
 * - `\u0300-\u0308` -> `<accents>`
 *
 * @param line - set of characters
 * @param rangeChar - the character to indicate ranges, set to empty to not have ranges.
 */
function expandCharacterSet(line, rangeChar = '-') {
    const charSet = new Set();
    let mode = 0;
    let prev = '';
    for (const char of line) {
        if (mode) {
            expandRange(prev, char).forEach((a) => charSet.add(a));
            mode = 0;
        }
        if (char === rangeChar) {
            // store the `-` if there isn't a previous value.
            if (prev) {
                mode = 1;
                continue;
            }
        }
        charSet.add(char);
        prev = char;
    }
    // catch the trailing `-`
    if (mode)
        charSet.add(rangeChar);
    return charSet;
}
exports.expandCharacterSet = expandCharacterSet;
/**
 * Expands a range between two characters.
 * - `a <= b` -- `[a, b]`
 * - `a > b` -- `[]`
 * @param a - staring character
 * @param b - ending character
 * @returns array of unicode characters.
 */
function expandRange(a, b) {
    const values = [];
    const end = b.codePointAt(0);
    const begin = a.codePointAt(0);
    if (!(begin && end))
        return values;
    for (let i = begin; i <= end; ++i) {
        values.push(String.fromCodePoint(i));
    }
    return values;
}
exports.expandRange = expandRange;
/**
 * Tries to find the different cases for a letter.
 * It can generate multiple forms:
 * - `ß` => `['ß', 'SS', 'ss']`
 * - `a` => `['a', 'A']`
 * - `A` => `['A', 'z']`
 * - `Å` => `['A', 'z']`
 * @param letter - the letter to generate upper and lower cases.
 * @param locale - the locale to use for changing case.
 * @returns the set of found cases.
 */
function caseForms(letter, locale) {
    const forms = new Set([letter]);
    function tryCases(s) {
        forms.add(s.toLocaleLowerCase(locale));
        forms.add(s.toLocaleUpperCase(locale));
    }
    tryCases(letter);
    [...forms].forEach(tryCases);
    return [...forms].filter((a) => !!a);
}
exports.caseForms = caseForms;
/**
 * Generate the different normalized forms of the letters.
 * @param letter - letter to normalize.
 * @returns combined set of possible forms.
 */
function accentForms(letter) {
    const forms = new Set([letter, letter.normalize('NFC'), letter.normalize('NFD')]);
    return forms;
}
exports.accentForms = accentForms;
/**
 * Remove all accents.
 * @param characters - unicode characters
 * @returns characters with accents removed (if it was possible)
 */
function stripAccents(characters) {
    return characters.normalize('NFD').replace(/\p{M}/gu, '');
}
exports.stripAccents = stripAccents;
/**
 * Remove all non accent characters from a string.
 * @param characters - characters with accents.
 * @returns - only the accents.
 */
function stripNonAccents(characters) {
    return characters.normalize('NFD').replace(/[^\p{M}]/gu, '');
}
exports.stripNonAccents = stripNonAccents;
//# sourceMappingURL=text.js.map