"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Trie = exports.FORBID = exports.NORMALIZED = exports.OPTIONAL_COMPOUND = exports.COMPOUND = exports.OPTIONAL_COMPOUND_FIX = exports.FORBID_PREFIX = exports.defaultTrieOptions = exports.COMPOUND_FIX = exports.CASE_INSENSITIVE_PREFIX = void 0;
const gensequence_1 = require("gensequence");
const constants_1 = require("./constants");
const find_1 = require("./find");
const suggest_1 = require("./suggest");
const trie_util_1 = require("./trie-util");
const util_1 = require("./utils/util");
const walker_1 = require("./walker");
var constants_2 = require("./constants");
Object.defineProperty(exports, "CASE_INSENSITIVE_PREFIX", { enumerable: true, get: function () { return constants_2.CASE_INSENSITIVE_PREFIX; } });
Object.defineProperty(exports, "COMPOUND_FIX", { enumerable: true, get: function () { return constants_2.COMPOUND_FIX; } });
Object.defineProperty(exports, "defaultTrieOptions", { enumerable: true, get: function () { return constants_2.defaultTrieOptions; } });
Object.defineProperty(exports, "FORBID_PREFIX", { enumerable: true, get: function () { return constants_2.FORBID_PREFIX; } });
Object.defineProperty(exports, "OPTIONAL_COMPOUND_FIX", { enumerable: true, get: function () { return constants_2.OPTIONAL_COMPOUND_FIX; } });
/** @deprecated */
exports.COMPOUND = constants_1.COMPOUND_FIX;
/** @deprecated */
exports.OPTIONAL_COMPOUND = constants_1.OPTIONAL_COMPOUND_FIX;
/** @deprecated */
exports.NORMALIZED = constants_1.CASE_INSENSITIVE_PREFIX;
/** @deprecated */
exports.FORBID = constants_1.FORBID_PREFIX;
const defaultLegacyMinCompoundLength = 3;
class Trie {
    constructor(root, count) {
        this.root = root;
        this.count = count;
        this._options = (0, trie_util_1.mergeOptionalWithDefaults)(root);
        this.isLegacy = this.calcIsLegacy();
        this.hasForbidden = !!root.c.get(root.forbiddenWordPrefix);
    }
    /**
     * Number of words in the Trie
     */
    size() {
        var _a;
        this.count = (_a = this.count) !== null && _a !== void 0 ? _a : (0, trie_util_1.countWords)(this.root);
        return this.count;
    }
    isSizeKnown() {
        return this.count !== undefined;
    }
    get options() {
        return this._options;
    }
    /**
     * @param text - text to find in the Trie
     * @param minCompoundLength - deprecated - allows words to be glued together
     */
    find(text, minCompoundLength = false) {
        const minLength = !minCompoundLength
            ? undefined
            : minCompoundLength === true
                ? defaultLegacyMinCompoundLength
                : minCompoundLength;
        const options = this.createFindOptions({
            compoundMode: minLength ? 'legacy' : 'compound',
            legacyMinCompoundLength: minLength,
        });
        return (0, find_1.findWordNode)(this.root, text, options).node;
    }
    /**
     *
     * @param text - text to search for
     * @param minCompoundLength - minimum word compound length
     * @deprecated - this method is no longer needed since compounding can be explicitly defined by the dictionary words.
     */
    findCompound(text, minCompoundLength = defaultLegacyMinCompoundLength) {
        const options = this.createFindOptions({ legacyMinCompoundLength: minCompoundLength });
        const r = (0, find_1.findLegacyCompound)(this.root, text, options);
        return r.node;
    }
    findExact(text) {
        const options = this.createFindOptions({ compoundMode: 'none' });
        return (0, find_1.findWordNode)(this.root, text, options).node;
    }
    has(word, minLegacyCompoundLength) {
        if (this.hasWord(word, false))
            return true;
        if (minLegacyCompoundLength) {
            const f = this.findWord(word, { useLegacyWordCompounds: minLegacyCompoundLength });
            return !!f.found;
        }
        return false;
    }
    /**
     * Determine if a word is in the dictionary.
     * @param word - the exact word to search for - must be normalized.
     * @param caseSensitive - false means also searching a dictionary where the words were normalized to lower case and accents removed.
     * @returns true if the word was found and is not forbidden.
     */
    hasWord(word, caseSensitive) {
        const f = this.findWord(word, { caseSensitive });
        return !!f.found && !f.forbidden;
    }
    findWord(word, options) {
        if (options === null || options === void 0 ? void 0 : options.useLegacyWordCompounds) {
            const len = options.useLegacyWordCompounds !== true
                ? options.useLegacyWordCompounds
                : defaultLegacyMinCompoundLength;
            const findOptions = this.createFindOptions({
                legacyMinCompoundLength: len,
                matchCase: options.caseSensitive,
            });
            return (0, find_1.findLegacyCompound)(this.root, word, findOptions);
        }
        const findOptions = this.createFindOptions({ matchCase: options === null || options === void 0 ? void 0 : options.caseSensitive });
        return (0, find_1.findWord)(this.root, word, findOptions);
    }
    /**
     * Determine if a word is in the forbidden word list.
     * @param word the word to lookup.
     */
    isForbiddenWord(word) {
        return this.hasForbidden && (0, find_1.isForbiddenWord)(this.root, word, this.options.forbiddenWordPrefix);
    }
    /**
     * Provides an ordered sequence of words with the prefix of text.
     */
    completeWord(text) {
        const n = this.find(text);
        const compoundChar = this.options.compoundCharacter;
        const subNodes = (0, trie_util_1.iteratorTrieWords)(n || {})
            .filter((w) => w[w.length - 1] !== compoundChar)
            .map((suffix) => text + suffix);
        return (0, gensequence_1.genSequence)(n && (0, trie_util_1.isWordTerminationNode)(n) ? [text] : []).concat(subNodes);
    }
    /**
     * Suggest spellings for `text`.  The results are sorted by edit distance with changes near the beginning of a word having a greater impact.
     * @param text - the text to search for
     * @param maxNumSuggestions - the maximum number of suggestions to return.
     * @param compoundMethod - Use to control splitting words.
     * @param numChanges - the maximum number of changes allowed to text. This is an approximate value, since some changes cost less than others.
     *                      the lower the value, the faster results are returned. Values less than 4 are best.
     */
    suggest(text, options) {
        return this.suggestWithCost(text, options).map((a) => a.word);
    }
    /**
     * Suggest spellings for `text`.  The results are sorted by edit distance with changes near the beginning of a word having a greater impact.
     * The results include the word and adjusted edit cost.  This is useful for merging results from multiple tries.
     */
    suggestWithCost(text, options) {
        const sep = options.compoundSeparator;
        const adjWord = sep ? (0, util_1.replaceAllFactory)(sep, '') : (a) => a;
        const optFilter = options.filter;
        const filter = optFilter
            ? (word, cost) => {
                const w = adjWord(word);
                return !this.isForbiddenWord(w) && optFilter(w, cost);
            }
            : (word) => !this.isForbiddenWord(adjWord(word));
        const opts = { ...options, filter };
        return (0, suggest_1.suggest)(this.root, text, opts);
    }
    /**
     * genSuggestions will generate suggestions and send them to `collector`. `collector` is responsible for returning the max acceptable cost.
     * Costs are measured in weighted changes. A cost of 100 is the same as 1 edit. Some edits are considered cheaper.
     * Returning a MaxCost < 0 will effectively cause the search for suggestions to stop.
     */
    genSuggestions(collector, compoundMethod) {
        const filter = (word) => !this.isForbiddenWord(word);
        const options = (0, trie_util_1.clean)({ compoundMethod, ...collector.genSuggestionOptions });
        const suggestions = (0, suggest_1.genSuggestions)(this.root, collector.word, options);
        collector.collect(suggestions, undefined, filter);
    }
    /**
     * Returns an iterator that can be used to get all words in the trie. For some dictionaries, this can result in millions of words.
     */
    words() {
        return (0, trie_util_1.iteratorTrieWords)(this.root);
    }
    /**
     * Allows iteration over the entire tree.
     * On the returned Iterator, calling .next(goDeeper: boolean), allows for controlling the depth.
     */
    iterate() {
        return (0, walker_1.walker)(this.root);
    }
    insert(word) {
        (0, trie_util_1.insert)(word, this.root);
        return this;
    }
    calcIsLegacy() {
        const c = this.root.c;
        return !((c === null || c === void 0 ? void 0 : c.get(this._options.compoundCharacter)) ||
            (c === null || c === void 0 ? void 0 : c.get(this._options.stripCaseAndAccentsPrefix)) ||
            (c === null || c === void 0 ? void 0 : c.get(this._options.forbiddenWordPrefix)));
    }
    static create(words, options) {
        const root = (0, trie_util_1.createTriFromList)(words, options);
        (0, trie_util_1.orderTrie)(root);
        return new Trie(root, undefined);
    }
    createFindOptions(options = {}) {
        const { caseInsensitivePrefix = this._options.stripCaseAndAccentsPrefix, compoundFix = this._options.compoundCharacter, forbidPrefix = this._options.forbiddenWordPrefix, } = (0, trie_util_1.clean)(options);
        const findOptions = (0, find_1.createFindOptions)({
            ...options,
            caseInsensitivePrefix,
            compoundFix,
            forbidPrefix,
        });
        return findOptions;
    }
}
exports.Trie = Trie;
//# sourceMappingURL=trie.js.map