import { WeightMap } from '..';
import { RequireOptional } from '../types';
import { GenSuggestionOptions, GenSuggestionOptionsStrict } from './genSuggestionsOptions';
export declare const DEFAULT_COMPOUNDED_WORD_SEPARATOR = "\u2219";
export declare type Cost = number;
export declare type MaxCost = Cost;
export interface SuggestionResultBase {
    /** The suggested word */
    word: string;
    /** The edit cost 100 = 1 edit */
    cost: Cost;
}
export interface SuggestionResult extends SuggestionResultBase {
    /** The suggested word with compound marks, generally a `•` */
    compoundWord?: string | undefined;
}
export interface Progress {
    type: 'progress';
    /** Number of Completed Tasks so far */
    completed: number;
    /**
     * Number of tasks remaining, this number is allowed to increase over time since
     * completed tasks can generate new tasks.
     */
    remaining: number;
}
export declare type GenerateNextParam = MaxCost | symbol | undefined;
export declare type GenerateSuggestionResult = SuggestionResultBase | Progress | undefined;
/**
 * Ask for the next result.
 * maxCost - sets the max cost for following suggestions
 * This is used to limit which suggestions are emitted.
 * If the `iterator.next()` returns `undefined`, it is to request a value for maxCost.
 *
 * The SuggestionIterator is generally the
 */
export declare type SuggestionGenerator = Generator<GenerateSuggestionResult, void, GenerateNextParam>;
export declare function compSuggestionResults(a: SuggestionResultBase, b: SuggestionResultBase): number;
export declare type FilterWordFn = (word: string, cost: number) => boolean;
export interface SuggestionCollector {
    /**
     * Collection suggestions from a SuggestionIterator
     * @param src - the SuggestionIterator used to generate suggestions.
     * @param timeout - the amount of time in milliseconds to allow for suggestions.
     * before sending `symbolStopProcessing`
     * Iterator implementation:
     * @example
     * r = yield(suggestion);
     * if (r === collector.symbolStopProcessing) // ...stop generating suggestions.
     */
    collect: (src: SuggestionGenerator, timeout?: number, filter?: FilterWordFn) => void;
    add: (suggestion: SuggestionResultBase) => SuggestionCollector;
    readonly suggestions: SuggestionResult[];
    readonly changeLimit: number;
    readonly maxCost: number;
    readonly word: string;
    readonly maxNumSuggestions: number;
    readonly includesTies: boolean;
    readonly ignoreCase: boolean;
    readonly genSuggestionOptions: GenSuggestionOptions;
    /**
     * Possible value sent to the SuggestionIterator telling it to stop processing.
     */
    readonly symbolStopProcessing: symbol;
}
export interface SuggestionCollectorOptions extends Omit<GenSuggestionOptionsStrict, 'ignoreCase' | 'changeLimit'> {
    /**
     * number of best matching suggestions.
     * @default 10
     */
    numSuggestions: number;
    /**
     * An optional filter function that can be used to limit remove unwanted suggestions.
     * I.E. to remove forbidden terms.
     * @default () => true
     */
    filter?: FilterWordFn;
    /**
     * The number of letters that can be changed when looking for a match
     * @default 5
     */
    changeLimit: number | undefined;
    /**
     * Include suggestions with tied cost even if the number is greater than `numSuggestions`.
     * @default true
     */
    includeTies?: boolean;
    /**
     * specify if case / accents should be ignored when looking for suggestions.
     * @default true
     */
    ignoreCase: boolean | undefined;
    /**
     * the total amount of time to allow for suggestions.
     * @default 1000
     */
    timeout?: number | undefined;
    /**
     * Used to improve the sorted results.
     */
    weightMap?: WeightMap | undefined;
}
export declare const defaultSuggestionCollectorOptions: RequireOptional<SuggestionCollectorOptions>;
export declare function suggestionCollector(wordToMatch: string, options: SuggestionCollectorOptions): SuggestionCollector;
/**
 * Impersonating a Collector, allows searching for multiple variants on the same word.
 * The collection is still in the original collector.
 * @param collector - collector to impersonate
 * @param word - word to present instead of `collector.word`.
 * @returns a SuggestionCollector
 */
export declare function impersonateCollector(collector: SuggestionCollector, word: string): SuggestionCollector;
export declare function isSuggestionResult(s: GenerateSuggestionResult): s is SuggestionResult;
//# sourceMappingURL=suggestCollector.d.ts.map