"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.genSuggestions = exports.suggest = exports.genCompoundableSuggestions = void 0;
const walker_1 = require("./walker");
const suggestCollector_1 = require("./suggestCollector");
const PairingHeap_1 = require("../utils/PairingHeap");
const orthography_1 = require("./orthography");
const genSuggestionsOptions_1 = require("./genSuggestionsOptions");
function* genCompoundableSuggestions(root, word, options) {
    const { compoundMethod, ignoreCase, changeLimit } = options;
    const len = word.length;
    const nodes = determineInitialNodes(root, ignoreCase);
    const noFollow = determineNoFollow(root);
    function compare(a, b) {
        const deltaCost = a.g - b.g;
        if (deltaCost)
            return deltaCost;
        // The costs are the some return the one with the most progress.
        return b.i - a.i;
    }
    const opCosts = {
        baseCost: 100,
        swapCost: 75,
        duplicateLetterCost: 25,
        visuallySimilar: 1,
        firstLetterBias: 25,
        wordBreak: 99,
    };
    const bc = opCosts.baseCost;
    const maxCostScale = 1.03 / 2;
    const mapSugCost = opCosts.visuallySimilar;
    const wordSeparator = compoundMethod === walker_1.CompoundWordsMethod.JOIN_WORDS ? walker_1.JOIN_SEPARATOR : walker_1.WORD_SEPARATOR;
    const compoundIndicator = root.compoundCharacter;
    let costLimit = bc * Math.min(len * maxCostScale, changeLimit);
    let stopNow = false;
    const candidates = new PairingHeap_1.PairingHeap(compare);
    const locationCache = new Map();
    const wordsToEmit = [];
    const pathToLocation = new Map();
    const edgesToResolve = [];
    const emittedWords = new Map();
    function updateCostLimit(maxCost) {
        switch (typeof maxCost) {
            case 'number':
                costLimit = Math.min(maxCost, costLimit);
                break;
            case 'symbol':
                stopNow = true;
                break;
        }
    }
    function getLocationNode(path) {
        const index = path.i;
        const node = path.n;
        const foundByIndex = locationCache.get(index);
        const byTrie = foundByIndex || new Map();
        if (!foundByIndex)
            locationCache.set(index, byTrie);
        const f = byTrie.get(node);
        const n = f || { in: new Map(), er: [], bc: 0, p: path, sbc: -1, sfx: [] };
        if (!f)
            byTrie.set(node, n);
        return n;
    }
    function* emitWord(word, cost) {
        if (cost <= costLimit) {
            // console.log(`e: ${word} ${cost}`);
            const f = emittedWords.get(word);
            if (f !== undefined && f <= cost)
                return undefined;
            emittedWords.set(word, cost);
            const lastChar = word[word.length - 1];
            if (!noFollow[lastChar]) {
                updateCostLimit(yield { word: word, cost: cost });
            }
        }
        return undefined;
    }
    function* emitWords() {
        for (const w of wordsToEmit) {
            yield* emitWord(w.word, w.cost);
        }
        wordsToEmit.length = 0;
        return undefined;
    }
    function addEdgeToBeResolved(path, edge) {
        path.r = path.r || new Set();
        path.r.add(edge);
    }
    function addEdge(path, edge) {
        const g = path.g + edge.c;
        const i = edge.i;
        if (g > costLimit)
            return undefined;
        const { n } = edge;
        const w = path.w + edge.s;
        const can = { e: edge, n, i, w, g, r: undefined, a: true };
        const location = getLocationNode(can);
        location.er.push(edge);
        // Is Location Resolved
        if (location.sbc >= 0 && location.sbc <= can.g) {
            // No need to go further, this node has been resolved.
            // Return the edge to be resolved
            addEdgeToBeResolved(path, edge);
            edgesToResolve.push({ edge, suffixes: location.sfx });
            return undefined;
        }
        const found = location.in.get(can.w);
        if (found) {
            // If the existing path is cheaper or the same keep it.
            // Do not add the edge.
            if (found.g <= can.g)
                return undefined;
            // Otherwise mark it as inactive and
            const e = found.e;
            if (e) {
                edgesToResolve.push({ edge: e, suffixes: [] });
            }
            found.a = false;
        }
        addEdgeToBeResolved(path, edge);
        location.in.set(can.w, can);
        if (location.p.g > can.g) {
            pathToLocation.delete(location.p);
            location.sbc = -1;
            location.sfx.length = 0;
            location.p = can;
        }
        if (location.p === can) {
            // Make this path the representation of this location.
            pathToLocation.set(can, location);
            candidates.add(can);
        }
        return edge;
    }
    function opWordFound(best) {
        if (!best.n.f)
            return;
        const i = best.i;
        const toDelete = len - i;
        const edge = { p: best, n: best.n, i, s: '', c: bc * toDelete, a: Action.Delete };
        addEdgeToBeResolved(best, edge);
        edgesToResolve.push({ edge, suffixes: [{ s: '', c: 0 }] });
        if (compoundMethod && best.g + opCosts.wordBreak <= costLimit) {
            const s = wordSeparator;
            nodes.forEach((node) => {
                const e = { p: best, n: node, i, s, c: opCosts.wordBreak, a: Action.WordBreak };
                addEdge(best, e);
            });
        }
    }
    function opCompoundWord(best) {
        var _a;
        if (!((_a = best.n.c) === null || _a === void 0 ? void 0 : _a.get(compoundIndicator)))
            return;
        const i = best.i;
        const s = '';
        nodes.forEach((node) => {
            var _a;
            const n = (_a = node.c) === null || _a === void 0 ? void 0 : _a.get(compoundIndicator);
            if (!n)
                return;
            const e = { p: best, n, i, s, c: opCosts.wordBreak, a: Action.CompoundWord };
            addEdge(best, e);
        });
    }
    function opInsert(best) {
        const children = best.n.c;
        if (!children)
            return;
        const i = best.i;
        const c = bc;
        for (const [s, n] of children) {
            const e = { p: best, n, i, s, c, a: Action.Insert };
            addEdge(best, e);
        }
    }
    function opDelete(best, num = 1) {
        const i = best.i;
        const e = {
            p: best,
            n: best.n,
            i: i + num,
            s: '',
            c: bc * num,
            a: Action.Delete,
            // t: word.slice(i, i + num),
        };
        return addEdge(best, e);
    }
    function opIdentity(best) {
        var _a;
        const s = word[best.i];
        const n = (_a = best.n.c) === null || _a === void 0 ? void 0 : _a.get(s);
        if (!n)
            return;
        const i = best.i + 1;
        const e = { p: best, n, i, s, c: 0, a: Action.Identity };
        addEdge(best, e);
    }
    function opReplace(best) {
        const children = best.n.c;
        if (!children)
            return;
        const wc = word[best.i];
        const wg = orthography_1.visualLetterMaskMap[wc] || 0;
        const i = best.i + 1;
        const cost = bc + (best.i ? 0 : opCosts.firstLetterBias);
        for (const [s, n] of children) {
            if (s == wc)
                continue;
            const sg = orthography_1.visualLetterMaskMap[s] || 0;
            const c = wg & sg ? mapSugCost : cost;
            const e = { p: best, n, i, s, c, a: Action.Replace /* , t: wc */ };
            addEdge(best, e);
        }
    }
    function opSwap(best) {
        var _a, _b;
        const children = best.n.c;
        const i = best.i;
        const i2 = i + 1;
        if (!children || len <= i2)
            return;
        const wc1 = word[i];
        const wc2 = word[i2];
        if (wc1 === wc2)
            return;
        const n = (_a = best.n.c) === null || _a === void 0 ? void 0 : _a.get(wc2);
        const n2 = (_b = n === null || n === void 0 ? void 0 : n.c) === null || _b === void 0 ? void 0 : _b.get(wc1);
        if (!n || !n2)
            return;
        const e = {
            p: best,
            n: n2,
            i: i2 + 1,
            s: wc2 + wc1,
            c: opCosts.swapCost,
            a: Action.Swap,
            // , t: wc1 + wc2,
        };
        addEdge(best, e);
    }
    function opDuplicate(best) {
        var _a, _b;
        const children = best.n.c;
        const i = best.i;
        const i2 = i + 1;
        if (!children || len <= i2)
            return;
        const wc1 = word[i];
        const wc2 = word[i2];
        const n = (_a = best.n.c) === null || _a === void 0 ? void 0 : _a.get(wc1);
        if (!n)
            return;
        if (wc1 === wc2) {
            // convert double letter to single
            const e = { p: best, n, i: i + 2, s: wc1, c: opCosts.duplicateLetterCost, a: Action.Delete };
            addEdge(best, e);
            return;
        }
        const n2 = (_b = n === null || n === void 0 ? void 0 : n.c) === null || _b === void 0 ? void 0 : _b.get(wc1);
        if (!n2)
            return;
        // convert single to double letter
        const e = { p: best, n: n2, i: i2, s: wc1 + wc1, c: opCosts.duplicateLetterCost, a: Action.Insert };
        addEdge(best, e);
    }
    function resolveEdges() {
        let e;
        while ((e = edgesToResolve.shift())) {
            resolveEdge(e);
        }
    }
    function resolveLocationEdges(location, suffixes) {
        for (const edge of location.er) {
            edgesToResolve.push({ edge, suffixes });
        }
    }
    function resolveEdge({ edge, suffixes }) {
        var _a;
        const { p, s: es, c: ec } = edge;
        if (!((_a = p.r) === null || _a === void 0 ? void 0 : _a.has(edge)))
            return;
        const edgeSuffixes = suffixes.map((sfx) => ({ s: es + sfx.s, c: ec + sfx.c }));
        for (const { s, c } of edgeSuffixes) {
            const cost = p.g + c;
            if (cost <= costLimit) {
                const word = p.w + s;
                wordsToEmit.push({ word, cost });
            }
        }
        p.r.delete(edge);
        const location = pathToLocation.get(p);
        if ((location === null || location === void 0 ? void 0 : location.p) === p) {
            location.sfx = location.sfx.concat(edgeSuffixes);
            if (!p.r.size) {
                location.sbc = p.g;
                resolveLocationEdges(location, edgeSuffixes);
            }
        }
        else if (!p.r.size) {
            if (p.e) {
                // Keep rolling up.
                edgesToResolve.push({ edge: p.e, suffixes: edgeSuffixes });
            }
        }
    }
    function cancelEdges(path) {
        if (!path.r)
            return;
        const suffixes = [];
        for (const edge of path.r) {
            edgesToResolve.push({ edge, suffixes });
        }
    }
    /************
     * Below is the core of the A* algorithm
     */
    updateCostLimit(yield undefined);
    nodes.forEach((node, idx) => {
        const g = idx ? 1 : 0;
        candidates.add({ e: undefined, n: node, i: 0, w: '', g, r: undefined, a: true });
    });
    const iterationsBeforePolling = 100;
    let i = iterationsBeforePolling;
    let maxSize = 0;
    let best;
    // const bc2 = 2 * bc;
    while (!stopNow && (best = candidates.dequeue())) {
        if (--i < 0) {
            i = iterationsBeforePolling;
            updateCostLimit(yield undefined);
        }
        maxSize = Math.max(maxSize, candidates.length);
        if (!best.a) {
            // best's edges are already part of a location node.
            continue;
        }
        if (best.g > costLimit) {
            cancelEdges(best);
            continue;
        }
        const bi = best.i;
        opWordFound(best);
        const children = best.n.c;
        if (!children)
            continue;
        if (bi === len) {
            opInsert(best);
        }
        else {
            opIdentity(best);
            opReplace(best);
            opDelete(best);
            opInsert(best);
            opSwap(best);
            opCompoundWord(best);
            opDuplicate(best);
        }
        resolveEdges();
        yield* emitWords();
    }
    resolveEdges();
    yield* emitWords();
    // console.log(`
    // word: ${word}
    // maxSize: ${maxSize}
    // length: ${candidates.length}
    // `);
    return undefined;
}
exports.genCompoundableSuggestions = genCompoundableSuggestions;
var Action;
(function (Action) {
    Action[Action["Identity"] = 0] = "Identity";
    Action[Action["Replace"] = 1] = "Replace";
    Action[Action["Delete"] = 2] = "Delete";
    Action[Action["Insert"] = 3] = "Insert";
    Action[Action["Swap"] = 4] = "Swap";
    Action[Action["CompoundWord"] = 5] = "CompoundWord";
    Action[Action["WordBreak"] = 6] = "WordBreak";
})(Action || (Action = {}));
function suggest(root, word, options) {
    const opts = (0, genSuggestionsOptions_1.createSuggestionOptions)(options);
    const collector = (0, suggestCollector_1.suggestionCollector)(word, {
        numSuggestions: opts.numSuggestions,
        changeLimit: opts.changeLimit,
        includeTies: opts.includeTies,
        ignoreCase: opts.ignoreCase,
        timeout: opts.timeout,
    });
    collector.collect(genSuggestions(root, word, opts));
    return collector.suggestions;
}
exports.suggest = suggest;
function* genSuggestions(root, word, options) {
    const roots = Array.isArray(root) ? root : [root];
    for (const r of roots) {
        yield* genCompoundableSuggestions(r, word, options);
    }
    return undefined;
}
exports.genSuggestions = genSuggestions;
function determineNoFollow(root) {
    const noFollow = Object.assign(Object.create(null), {
        [root.compoundCharacter]: true,
        [root.forbiddenWordPrefix]: true,
        [root.stripCaseAndAccentsPrefix]: true,
    });
    return noFollow;
}
function determineInitialNodes(root, ignoreCase) {
    const roots = Array.isArray(root) ? root : [root];
    const rootNodes = roots;
    const noCaseNodes = ignoreCase
        ? roots
            .filter((r) => r.stripCaseAndAccentsPrefix)
            .map((n) => { var _a; return (_a = n.c) === null || _a === void 0 ? void 0 : _a.get(n.stripCaseAndAccentsPrefix); })
            .filter(isDefined)
        : [];
    const nodes = rootNodes.concat(noCaseNodes);
    return nodes;
}
function isDefined(v) {
    return v !== undefined;
}
//# sourceMappingURL=suggestAStar.js.map