"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.__testing__ = exports.createFindOptions = exports.isForbiddenWord = exports.isEndOfWordNode = exports.findWordExact = exports.findCompoundNode = exports.findLegacyCompound = exports.findWordNode = exports.findWord = void 0;
const constants_1 = require("./constants");
const trie_util_1 = require("./trie-util");
const TrieNode_1 = require("./TrieNode");
const defaultLegacyMinCompoundLength = 3;
const _defaultFindOptions = {
    matchCase: false,
    compoundMode: 'compound',
    forbidPrefix: constants_1.FORBID_PREFIX,
    compoundFix: constants_1.COMPOUND_FIX,
    caseInsensitivePrefix: constants_1.CASE_INSENSITIVE_PREFIX,
    legacyMinCompoundLength: defaultLegacyMinCompoundLength,
};
const arrayCompoundModes = ['none', 'compound', 'legacy'];
const knownCompoundModes = new Map(arrayCompoundModes.map((a) => [a, a]));
/**
 *
 * @param root Trie root node. root.c contains the compound root and forbidden root.
 * @param word A pre normalized word use `normalizeWord` or `normalizeWordToLowercase`
 * @param options
 */
function findWord(root, word, options) {
    return _findWord(root, word, createFindOptions(options));
}
exports.findWord = findWord;
/**
 *
 * @param root Trie root node. root.c contains the compound root and forbidden root.
 * @param word A pre normalized word use `normalizeWord` or `normalizeWordToLowercase`
 * @param options
 */
function findWordNode(root, word, options) {
    return _findWordNode(root, word, createFindOptions(options));
}
exports.findWordNode = findWordNode;
/**
 *
 * @param root Trie root node. root.c contains the compound root and forbidden root.
 * @param word A pre normalized word use `normalizeWord` or `normalizeWordToLowercase`
 * @param options
 */
function _findWord(root, word, options) {
    const { node: _, ...result } = _findWordNode(root, word, options);
    return result;
}
/**
 *
 * @param root Trie root node. root.c contains the compound root and forbidden root.
 * @param word A pre normalized word use `normalizeWord` or `normalizeWordToLowercase`
 * @param options
 */
function _findWordNode(root, word, options) {
    var _a, _b;
    const compoundMode = knownCompoundModes.get(options.compoundMode) || _defaultFindOptions.compoundMode;
    const compoundPrefix = options.compoundMode === 'compound' ? (_a = root.compoundCharacter) !== null && _a !== void 0 ? _a : options.compoundFix : '';
    const ignoreCasePrefix = options.matchCase ? '' : (_b = root.stripCaseAndAccentsPrefix) !== null && _b !== void 0 ? _b : options.caseInsensitivePrefix;
    function __findCompound() {
        const f = findCompoundWord(root, word, compoundPrefix, ignoreCasePrefix);
        const result = { ...f };
        if (f.found !== false && f.compoundUsed) {
            // If case was ignored when searching for the word, then check the forbidden
            // in the ignore case forbidden list.
            const r = !f.caseMatched ? walk(root, options.caseInsensitivePrefix) : root;
            result.forbidden = isForbiddenWord(r, word, options.forbidPrefix);
        }
        return result;
    }
    function __findExact() {
        const n = walk(root, word);
        const isFound = isEndOfWordNode(n);
        const result = {
            found: isFound && word,
            compoundUsed: false,
            forbidden: isForbiddenWord(root, word, options.forbidPrefix),
            node: n,
            caseMatched: true,
        };
        return result;
    }
    switch (compoundMode) {
        case 'none':
            return options.matchCase ? __findExact() : __findCompound();
        case 'compound':
            return __findCompound();
        case 'legacy':
            return findLegacyCompound(root, word, options);
    }
}
function findLegacyCompound(root, word, options) {
    const roots = [root];
    if (!options.matchCase) {
        roots.push(walk(root, options.caseInsensitivePrefix));
    }
    return findLegacyCompoundNode(roots, word, options.legacyMinCompoundLength);
}
exports.findLegacyCompound = findLegacyCompound;
function findCompoundNode(root, word, compoundCharacter, ignoreCasePrefix) {
    var _a, _b, _c;
    // Approach - do a depth first search for the matching word.
    const stack = [
        // { n: root, compoundPrefix: '', cr: undefined, caseMatched: true },
        { n: root, compoundPrefix: ignoreCasePrefix, cr: undefined, caseMatched: true },
    ];
    const compoundPrefix = compoundCharacter || ignoreCasePrefix;
    const possibleCompoundPrefix = ignoreCasePrefix && compoundCharacter ? ignoreCasePrefix + compoundCharacter : '';
    const w = word.normalize();
    function determineRoot(s) {
        var _a;
        const prefix = s.compoundPrefix;
        let r = root;
        let i;
        for (i = 0; i < prefix.length && r; ++i) {
            r = (_a = r.c) === null || _a === void 0 ? void 0 : _a.get(prefix[i]);
        }
        const caseMatched = s.caseMatched && prefix[0] !== ignoreCasePrefix;
        return {
            n: s.n,
            compoundPrefix: prefix === compoundPrefix ? possibleCompoundPrefix : '',
            cr: r,
            caseMatched,
        };
    }
    let compoundUsed = false;
    let caseMatched = true;
    let i = 0;
    let node;
    // eslint-disable-next-line no-constant-condition
    while (true) {
        const s = stack[i];
        const h = w[i++];
        const n = s.cr || s.n;
        const c = (_a = n === null || n === void 0 ? void 0 : n.c) === null || _a === void 0 ? void 0 : _a.get(h);
        if (c && i < word.length) {
            // Go deeper.
            caseMatched = s.caseMatched;
            stack[i] = { n: c, compoundPrefix, cr: undefined, caseMatched };
        }
        else if (!c || !c.f) {
            // Remember the first matching node for possible auto completion.
            node = node || c;
            // We did not find the word backup and take the first unused compound branch
            while (--i > 0 && (!stack[i].compoundPrefix || !((_c = (_b = stack[i].n) === null || _b === void 0 ? void 0 : _b.c) === null || _c === void 0 ? void 0 : _c.has(compoundCharacter)))) {
                /* empty */
            }
            if (i >= 0 && stack[i].compoundPrefix) {
                compoundUsed = i > 0;
                const r = determineRoot(stack[i]);
                stack[i] = r;
                if (!r.cr) {
                    break;
                }
                if (!i && !r.caseMatched) {
                    if (w !== w.toLowerCase()) {
                        // It is not going to be found.
                        break;
                    }
                }
            }
            else {
                break;
            }
        }
        else {
            node = c;
            caseMatched = s.caseMatched;
            break;
        }
    }
    const found = (i && i === word.length && word) || false;
    const result = { found, compoundUsed, node, forbidden: undefined, caseMatched };
    return result;
}
exports.findCompoundNode = findCompoundNode;
function findCompoundWord(root, word, compoundCharacter, ignoreCasePrefix) {
    const { found, compoundUsed, node, caseMatched } = findCompoundNode(root, word, compoundCharacter, ignoreCasePrefix);
    // Was it a word?
    if (!node || !node.f) {
        return { found: false, compoundUsed, node, forbidden: undefined, caseMatched };
    }
    return { found, compoundUsed, node, forbidden: undefined, caseMatched };
}
function findWordExact(root, word) {
    return isEndOfWordNode(walk(root, word));
}
exports.findWordExact = findWordExact;
function isEndOfWordNode(n) {
    return (n === null || n === void 0 ? void 0 : n.f) === TrieNode_1.FLAG_WORD;
}
exports.isEndOfWordNode = isEndOfWordNode;
function walk(root, word) {
    var _a;
    const w = word;
    let n = root;
    let i = 0;
    while (n && i < word.length) {
        const h = w[i++];
        n = (_a = n.c) === null || _a === void 0 ? void 0 : _a.get(h);
    }
    return n;
}
function findLegacyCompoundNode(roots, word, minCompoundLength) {
    var _a, _b;
    const root = roots[0];
    const numRoots = roots.length;
    // Approach - do a depth first search for the matching word.
    const stack = [
        { n: root, usedRoots: 1, subLength: 0, isCompound: false, cr: undefined, caseMatched: true },
    ];
    const w = word;
    const wLen = w.length;
    let compoundUsed = false;
    let caseMatched = true;
    let i = 0;
    let node;
    // eslint-disable-next-line no-constant-condition
    while (true) {
        const s = stack[i];
        const h = w[i++];
        const n = s.cr || s.n;
        const c = (_a = n === null || n === void 0 ? void 0 : n.c) === null || _a === void 0 ? void 0 : _a.get(h);
        if (c && i < wLen) {
            // Go deeper.
            stack[i] = {
                n: c,
                usedRoots: 0,
                subLength: s.subLength + 1,
                isCompound: s.isCompound,
                cr: undefined,
                caseMatched: s.caseMatched,
            };
        }
        else if (!c || !c.f || (c.f && s.subLength < minCompoundLength - 1)) {
            // We did not find the word backup and take the first unused compound branch
            while (--i > 0) {
                const s = stack[i];
                if (s.usedRoots < numRoots &&
                    ((_b = s.n) === null || _b === void 0 ? void 0 : _b.f) &&
                    (s.subLength >= minCompoundLength || !s.subLength) &&
                    wLen - i >= minCompoundLength) {
                    break;
                }
            }
            if (i > 0 || stack[i].usedRoots < numRoots) {
                compoundUsed = i > 0;
                const s = stack[i];
                s.cr = roots[s.usedRoots++];
                s.subLength = 0;
                s.isCompound = compoundUsed;
                s.caseMatched = s.caseMatched && s.usedRoots <= 1;
            }
            else {
                break;
            }
        }
        else {
            node = c;
            caseMatched = s.caseMatched;
            break;
        }
    }
    function extractWord() {
        if (!word || i < word.length)
            return false;
        const letters = [];
        let subLen = 0;
        for (let j = 0; j < i; ++j) {
            const { subLength } = stack[j];
            if (subLength < subLen) {
                letters.push('+');
            }
            letters.push(word[j]);
            subLen = subLength;
        }
        return letters.join('');
    }
    const found = extractWord();
    const result = { found, compoundUsed, node, forbidden: undefined, caseMatched };
    return result;
}
function findLegacyCompoundWord(roots, word, minCompoundLength) {
    const { found, compoundUsed, caseMatched } = findLegacyCompoundNode(roots, word, minCompoundLength);
    return { found, compoundUsed, caseMatched };
}
function isForbiddenWord(root, word, forbiddenPrefix) {
    var _a;
    return findWordExact((_a = root === null || root === void 0 ? void 0 : root.c) === null || _a === void 0 ? void 0 : _a.get(forbiddenPrefix), word);
}
exports.isForbiddenWord = isForbiddenWord;
function createFindOptions(options) {
    return (0, trie_util_1.mergeDefaults)(options, _defaultFindOptions);
}
exports.createFindOptions = createFindOptions;
exports.__testing__ = {
    findLegacyCompoundWord,
};
//# sourceMappingURL=find.js.map