"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.__testing__ = exports.lookupReplaceCost = exports.prettyPrintWeightMap = exports.prettyPrintSwap = exports.prettyPrintReplace = exports.splitMapSubstrings = exports.splitMapSubstringsIterable = exports.addDefsToWeightMap = exports.addAdjustment = exports.addDefToWeightMap = exports.createWeightMap = void 0;
const suggestCollector_1 = require("../suggestions/suggestCollector");
const matchPossibleWordSeparators = /[+∙•・●]/g;
function createWeightMap(...defs) {
    const map = _createWeightMap();
    addDefsToWeightMap(map, defs);
    return map;
}
exports.createWeightMap = createWeightMap;
function addDefToWeightMap(map, ...defs) {
    return addDefsToWeightMap(map, defs);
}
exports.addDefToWeightMap = addDefToWeightMap;
function addAdjustment(map, ...adjustments) {
    for (const adj of adjustments) {
        map.adjustments.set(adj.id, adj);
    }
    return map;
}
exports.addAdjustment = addAdjustment;
function addDefsToWeightMap(map, defs) {
    function addSet(set, def) {
        addSetToTrieCost(map.insDel, set, def.insDel, def.penalty);
        addSetToTrieTrieCost(map.replace, set, def.replace, def.penalty);
        addSetToTrieTrieCost(map.swap, set, def.swap, def.penalty);
    }
    for (const _def of defs) {
        const def = normalizeDef(_def);
        const mapSets = splitMap(def);
        mapSets.forEach((s) => addSet(s, def));
    }
    return map;
}
exports.addDefsToWeightMap = addDefsToWeightMap;
function _createWeightMap() {
    return new _WeightedMap();
}
function lowest(a, b) {
    if (a === undefined)
        return b;
    if (b === undefined)
        return a;
    return a <= b ? a : b;
}
function highest(a, b) {
    if (a === undefined)
        return b;
    if (b === undefined)
        return a;
    return a >= b ? a : b;
}
function normalize(s) {
    const f = new Set([s]);
    f.add(s.normalize('NFC'));
    f.add(s.normalize('NFD'));
    return f;
}
function* splitMapSubstringsIterable(map) {
    let seq = '';
    let mode = 0;
    for (const char of map) {
        if (mode && char === ')') {
            yield* normalize(seq);
            mode = 0;
            continue;
        }
        if (mode) {
            seq += char;
            continue;
        }
        if (char === '(') {
            mode = 1;
            seq = '';
            continue;
        }
        yield* normalize(char);
    }
}
exports.splitMapSubstringsIterable = splitMapSubstringsIterable;
function splitMapSubstrings(map) {
    return [...splitMapSubstringsIterable(map)];
}
exports.splitMapSubstrings = splitMapSubstrings;
/**
 * Splits a WeightedMapDef.map
 * @param map
 */
function splitMap(def) {
    const { map } = def;
    const sets = map.split('|');
    return sets.map(splitMapSubstrings).filter((s) => s.length > 0);
}
function addToTrieCost(trie, str, cost, penalties) {
    if (!str)
        return;
    let t = trie;
    for (const c of str) {
        const n = (t.n = t.n || Object.create(null));
        t = n[c] = n[c] || Object.create(null);
    }
    t.c = lowest(t.c, cost);
    t.p = highest(t.p, penalties);
}
function addToTrieTrieCost(trie, left, right, cost, penalties) {
    let t = trie;
    for (const c of left) {
        const n = (t.n = t.n || Object.create(null));
        t = n[c] = n[c] || Object.create(null);
    }
    const trieCost = (t.t = t.t || Object.create(null));
    addToTrieCost(trieCost, right, cost, penalties);
}
function addSetToTrieCost(trie, set, cost, penalties) {
    if (cost === undefined)
        return;
    for (const str of set) {
        addToTrieCost(trie, str, cost, penalties);
    }
}
function addSetToTrieTrieCost(trie, set, cost, penalties) {
    if (cost === undefined)
        return;
    for (const left of set) {
        for (const right of set) {
            if (left === right)
                continue;
            addToTrieTrieCost(trie, left, right, cost, penalties);
        }
    }
}
function* searchTrieNodes(trie, str, i) {
    const len = str.length;
    for (let n = trie.n; i < len && n;) {
        const t = n[str[i]];
        if (!t)
            return;
        ++i;
        yield { i, t };
        n = t.n;
    }
}
function* walkTrieNodes(t, s) {
    if (!t)
        return;
    yield { s, t };
    if (!t.n)
        return;
    for (const [k, v] of Object.entries(t.n)) {
        yield* walkTrieNodes(v, s + k);
    }
}
function* walkTrieCost(trie) {
    for (const { s, t } of walkTrieNodes(trie, '')) {
        if (t.c) {
            yield { s, c: t.c, p: t.p };
        }
    }
}
function* walkTrieTrieCost(trie) {
    for (const { s: a, t } of walkTrieNodes(trie, '')) {
        if (t.t) {
            for (const { s: b, c, p } of walkTrieCost(t.t)) {
                yield { a, b, c, p };
            }
        }
    }
}
function* findTrieCostPrefixes(trie, str, i) {
    for (const n of searchTrieNodes(trie, str, i)) {
        const { c, p } = n.t;
        if (c !== undefined) {
            yield { i: n.i, c, p: p || 0 };
        }
    }
}
function* findTrieTrieCostPrefixes(trie, str, i) {
    for (const n of searchTrieNodes(trie, str, i)) {
        const t = n.t.t;
        if (t !== undefined) {
            yield { i: n.i, t };
        }
    }
}
class _WeightedMap {
    constructor() {
        this.insDel = {};
        this.replace = {};
        this.swap = {};
        this.adjustments = new Map();
    }
    *calcInsDelCosts(pos) {
        const { a, ai, b, bi, c, p } = pos;
        for (const del of findTrieCostPrefixes(this.insDel, a, ai)) {
            yield { a, b, ai: del.i, bi, c: c + del.c, p: p + del.p };
        }
        for (const ins of findTrieCostPrefixes(this.insDel, b, bi)) {
            yield { a, b, ai, bi: ins.i, c: c + ins.c, p: p + ins.p };
        }
    }
    *calcReplaceCosts(pos) {
        // Search for matching substrings in `a` to be replaced by
        // matching substrings from `b`. All substrings start at their
        // respective `ai`/`bi` positions.
        const { a, ai, b, bi, c, p } = pos;
        for (const del of findTrieTrieCostPrefixes(this.replace, a, ai)) {
            for (const ins of findTrieCostPrefixes(del.t, b, bi)) {
                yield { a, b, ai: del.i, bi: ins.i, c: c + ins.c, p: p + ins.p };
            }
        }
    }
    *calcSwapCosts(pos) {
        const { a, ai, b, bi, c, p } = pos;
        const swap = this.swap;
        for (const left of findTrieTrieCostPrefixes(swap, a, ai)) {
            for (const right of findTrieCostPrefixes(left.t, a, left.i)) {
                const sw = a.slice(left.i, right.i) + a.slice(ai, left.i);
                if (b.slice(bi).startsWith(sw)) {
                    const len = sw.length;
                    yield { a, b, ai: ai + len, bi: bi + len, c: c + right.c, p: p + right.p };
                }
            }
        }
    }
    calcAdjustment(word) {
        let penalty = 0;
        for (const adj of this.adjustments.values()) {
            if (adj.regexp.global) {
                for (const _m of word.matchAll(adj.regexp)) {
                    penalty += adj.penalty;
                }
            }
            else if (adj.regexp.test(word)) {
                penalty += adj.penalty;
            }
        }
        return penalty;
    }
}
function prettyPrintInsDel(trie, pfx = '', indent = '  ') {
    function* walk() {
        for (const { s, c, p } of walkTrieCost(trie)) {
            const pm = p ? ` + ${p}` : '';
            yield indent + `(${s}) = ${c}${pm}`;
        }
    }
    return ['InsDel:', ...[...walk()].sort()].map((line) => pfx + line + '\n').join('');
}
function prettyPrintReplace(trie, pfx = '', indent = '  ') {
    function* walk() {
        for (const { a, b, c, p } of walkTrieTrieCost(trie)) {
            const pm = p ? ` + ${p}` : '';
            yield indent + `(${a}) -> (${b}) = ${c}${pm}`;
        }
    }
    return ['Replace:', ...[...walk()].sort()].map((line) => pfx + line + '\n').join('');
}
exports.prettyPrintReplace = prettyPrintReplace;
function prettyPrintSwap(trie, pfx = '', indent = '  ') {
    function* walk() {
        for (const { a, b, c, p } of walkTrieTrieCost(trie)) {
            const pm = p ? ` + ${p}` : '';
            yield indent + `(${a}) <-> (${b}) = ${c}${pm}`;
        }
    }
    return ['Swap:', ...[...walk()].sort()].map((line) => pfx + line + '\n').join('');
}
exports.prettyPrintSwap = prettyPrintSwap;
function prettyPrintWeightMap(map) {
    return [prettyPrintInsDel(map.insDel), prettyPrintReplace(map.replace), prettyPrintSwap(map.swap)].join('\n');
}
exports.prettyPrintWeightMap = prettyPrintWeightMap;
function lookupReplaceCost(map, a, b) {
    var _a, _b;
    const trie = map.replace;
    let tt = trie;
    for (let ai = 0; ai < a.length && tt; ++ai) {
        tt = (_a = tt.n) === null || _a === void 0 ? void 0 : _a[a[ai]];
    }
    if (!tt)
        return undefined;
    let t = tt.t;
    for (let bi = 0; bi < b.length && t; ++bi) {
        t = (_b = t.n) === null || _b === void 0 ? void 0 : _b[b[bi]];
    }
    return t === null || t === void 0 ? void 0 : t.c;
}
exports.lookupReplaceCost = lookupReplaceCost;
function normalizeDef(def) {
    const { map, ...rest } = def;
    return { ...rest, map: normalizeMap(map) };
}
function normalizeMap(map) {
    return map.replace(matchPossibleWordSeparators, suggestCollector_1.DEFAULT_COMPOUNDED_WORD_SEPARATOR);
}
exports.__testing__ = {
    findTrieCostPrefixes,
    findTrieTrieCostPrefixes,
    normalizeDef,
    splitMap,
    splitMapSubstrings,
};
//# sourceMappingURL=weightedMaps.js.map