import { Point2, Vec2 } from '../Vec2';
import BezierJSWrapper from './BezierJSWrapper';
import Rect2 from './Rect2';
/**
 * Represents a 2D [Bézier curve](https://en.wikipedia.org/wiki/B%C3%A9zier_curve).
 *
 * Example:
 * ```ts,runnable,console
 * import { QuadraticBezier, Vec2 } from '@js-draw/math';
 *
 * const startPoint = Vec2.of(4, 3);
 * const controlPoint = Vec2.of(1, 1);
 * const endPoint = Vec2.of(1, 3);
 *
 * const curve = new QuadraticBezier(
 *   startPoint,
 *   controlPoint,
 *   endPoint,
 * );
 *
 * console.log('Curve:', curve);
 * ```
 *
 * **Note**: Some Bézier operations internally use the `bezier-js` library.
 */
export declare class QuadraticBezier extends BezierJSWrapper {
    readonly p0: Point2;
    readonly p1: Point2;
    readonly p2: Point2;
    constructor(p0: Point2, p1: Point2, p2: Point2);
    /**
     * Returns a component of a quadratic Bézier curve at t, where p0,p1,p2 are either all x or
     * all y components of the target curve.
     */
    private static componentAt;
    private static derivativeComponentAt;
    private static secondDerivativeComponentAt;
    /**
     * @returns the curve evaluated at `t`.
     *
     * `t` should be a number in `[0, 1]`.
     */
    at(t: number): Point2;
    derivativeAt(t: number): Point2;
    secondDerivativeAt(t: number): Point2;
    normal(t: number): Vec2;
    /** @returns an overestimate of this shape's bounding box. */
    getLooseBoundingBox(): Rect2;
    /**
     * @returns the *approximate* distance from `point` to this curve.
     */
    approximateDistance(point: Point2): number;
    getPoints(): import("../Vec3").Vec3[];
}
export default QuadraticBezier;
