import Mat33 from '../Mat33';
import Rect2 from './Rect2';
import { Vec2, Point2 } from '../Vec2';
import Parameterized2DShape from './Parameterized2DShape';
import Vec3 from '../Vec3';
interface IntersectionResult {
    point: Point2;
    t: number;
}
/**
 * Represents a line segment. A `LineSegment2` is immutable.
 *
 * @example
 * ```ts,runnable,console
 * import {LineSegment2, Vec2} from '@js-draw/math';
 * const l = new LineSegment2(Vec2.of(1, 1), Vec2.of(2, 2));
 * console.log('length: ', l.length);
 * console.log('direction: ', l.direction);
 * console.log('bounding box: ', l.bbox);
 * ```
 */
export declare class LineSegment2 extends Parameterized2DShape {
    private readonly point1;
    private readonly point2;
    /**
     * The **unit** direction vector of this line segment, from
     * `point1` to `point2`.
     *
     * In other words, `direction` is `point2.minus(point1).normalized()`
     * (perhaps except when `point1` is equal to `point2`).
     */
    readonly direction: Vec2;
    /** The distance between `point1` and `point2`. */
    readonly length: number;
    /** The bounding box of this line segment. */
    readonly bbox: Rect2;
    /** Creates a new `LineSegment2` from its endpoints. */
    constructor(point1: Point2, point2: Point2);
    /**
     * Returns the smallest line segment that contains all points in `points`, or `null`
     * if no such line segment exists.
     *
     * @example
     * ```ts,runnable,console
     * import {LineSegment2, Vec2} from '@js-draw/math';
     * console.log(LineSegment2.ofSmallestContainingPoints([Vec2.of(1, 0), Vec2.of(0, 1)]));
     * ```
     */
    static ofSmallestContainingPoints(points: readonly Point2[]): LineSegment2 | null;
    /** Alias for `point1`. */
    get p1(): Point2;
    /** Alias for `point2`. */
    get p2(): Point2;
    get center(): Point2;
    /**
     * Gets a point a **distance** `t` along this line.
     *
     * @deprecated
     */
    get(t: number): Point2;
    /**
     * Returns a point a fraction, `t`, along this line segment.
     * Thus, `segment.at(0)` returns `segment.p1` and `segment.at(1)` returns
     * `segment.p2`.
     *
     * `t` should be in `[0, 1]`.
     */
    at(t: number): Point2;
    normalAt(_t: number): Vec2;
    tangentAt(_t: number): Vec3;
    splitAt(t: number): [LineSegment2] | [LineSegment2, LineSegment2];
    /**
     * Returns the intersection of this with another line segment.
     *
     * **WARNING**: The parameter value returned by this method does not range from 0 to 1 and
     *              is currently a length.
     *              This will change in a future release.
     * @deprecated
     */
    intersection(other: LineSegment2): IntersectionResult | null;
    intersects(other: LineSegment2): boolean;
    argIntersectsLineSegment(lineSegment: LineSegment2): number[];
    /**
     * Returns the points at which this line segment intersects the
     * given line segment.
     *
     * Note that {@link intersects} returns *whether* this line segment intersects another
     * line segment. This method, by contrast, returns **the point** at which the intersection
     * occurs, if such a point exists.
     */
    intersectsLineSegment(lineSegment: LineSegment2): Vec3[];
    closestPointTo(target: Point2): Vec3;
    nearestPointTo(target: Vec3): {
        point: Vec3;
        parameterValue: number;
    };
    /**
     * Returns the distance from this line segment to `target`.
     *
     * Because a line segment has no interior, this signed distance is equivalent to
     * the full distance between `target` and this line segment.
     */
    signedDistance(target: Point2): number;
    /** Returns a copy of this line segment transformed by the given `affineTransfm`. */
    transformedBy(affineTransfm: Mat33): LineSegment2;
    /** @inheritdoc */
    getTightBoundingBox(): Rect2;
    toString(): string;
    /**
     * Returns `true` iff this is equivalent to `other`.
     *
     * **Options**:
     * - `tolerance`: The maximum difference between endpoints. (Default: 0)
     * - `ignoreDirection`: Allow matching a version of `this` with opposite direction. (Default: `true`)
     */
    eq(other: LineSegment2, options?: {
        tolerance?: number;
        ignoreDirection?: boolean;
    }): boolean;
}
export default LineSegment2;
