import LineSegment2 from './LineSegment2';
import Mat33 from '../Mat33';
import { Point2, Vec2 } from '../Vec2';
import Abstract2DShape from './Abstract2DShape';
import Vec3 from '../Vec3';
/** An object that can be converted to a {@link Rect2}. */
export interface RectTemplate {
    x: number;
    y: number;
    w?: number;
    h?: number;
    width?: number;
    height?: number;
}
/**
 * Represents a rectangle in 2D space, parallel to the XY axes.
 *
 * **Example**:
 * ```ts,runnable,console
 * import { Rect2, Vec2 } from '@js-draw/math';
 *
 * const rect = Rect2.fromCorners(
 *   Vec2.of(0, 0),
 *   Vec2.of(10, 10),
 * );
 * console.log('area', rect.area);
 * console.log('topLeft', rect.topLeft);
 * ```
 *
 * `invariant: w ≥ 0, h ≥ 0, immutable`
 */
export declare class Rect2 extends Abstract2DShape {
    readonly x: number;
    readonly y: number;
    readonly w: number;
    readonly h: number;
    readonly topLeft: Point2;
    readonly size: Vec2;
    readonly area: number;
    constructor(x: number, y: number, w: number, h: number);
    translatedBy(vec: Vec2): Rect2;
    resizedTo(size: Vec2): Rect2;
    containsPoint(other: Point2): boolean;
    /** @returns true iff `other` is completely within this `Rect2`. */
    containsRect(other: Rect2): boolean;
    /**
     * @returns true iff this and `other` overlap
     */
    intersects(other: Rect2): boolean;
    intersection(other: Rect2): Rect2 | null;
    union(other: Rect2): Rect2;
    divideIntoGrid(columns: number, rows: number): Rect2[];
    grownToPoint(point: Point2, margin?: number): Rect2;
    grownBy(margin: number): Rect2;
    /**
     * If this rectangle is smaller than `minSize`, returns a copy of this
     * with a larger width/height.
     *
     * If smaller than `minSize`, padding is applied on both sides.
     */
    grownToSize(minSize: Vec2): Rect2;
    getClosestPointOnBoundaryTo(target: Point2): Vec3;
    /**
     * Returns `true` iff all points in this rectangle are within `distance` from `point`:
     *
     * If $R$ is the set of points in this rectangle, returns `true` iff
     * $$
     * 	\forall {\bf a} \in R, \|\texttt{point} - {\bf a}\| < \texttt{radius}
     * $$
     */
    isWithinRadiusOf(radius: number, point: Point2): boolean;
    get corners(): Point2[];
    get maxDimension(): number;
    get minDimension(): number;
    get bottomRight(): Vec3;
    get topRight(): Vec3;
    get bottomLeft(): Vec3;
    get width(): number;
    get height(): number;
    get center(): {
        readonly x: number;
        readonly y: number;
        readonly z: number;
        readonly xy: {
            x: number;
            y: number;
        };
        at(idx: number): number;
        length(): number;
        magnitude(): number;
        magnitudeSquared(): number;
        squareDistanceTo(p: Vec3): number;
        distanceTo(p: Vec3): number;
        maximumEntryMagnitude(): number;
        angle(): number;
        normalized(): Vec3;
        normalizedOrZero(): Vec3;
        times(c: number): Vec3;
        plus(v: Vec3): Vec3;
        minus(v: Vec3): Vec3;
        dot(other: Vec3): number;
        cross(other: Vec3): Vec3;
        scale(other: Vec3 | number): Vec3;
        orthog(): Vec3;
        extend(distance: number, direction: Vec3): Vec3;
        lerp(target: Vec3, fractionTo: number): Vec3;
        zip(other: Vec3, zip: (componentInThis: number, componentInOther: number) => number): Vec3;
        map(fn: (component: number, index: number) => number): Vec3;
        asArray(): [number, number, number];
        eq(other: Vec3, fuzz?: number): boolean;
        toString(): string;
    };
    getEdges(): LineSegment2[];
    intersectsLineSegment(lineSegment: LineSegment2): Point2[];
    signedDistance(point: Vec3): number;
    getTightBoundingBox(): Rect2;
    transformedBoundingBox(affineTransform: Mat33): Rect2;
    /** @return true iff this is equal to `other ± tolerance` */
    eq(other: Rect2, tolerance?: number): boolean;
    toString(): string;
    static fromCorners(corner1: Point2, corner2: Point2): Rect2;
    static bboxOf(points: Point2[], margin?: number): Rect2;
    static union(...rects: Rect2[]): Rect2;
    static of(template: RectTemplate): Rect2;
    static empty: Rect2;
    static unitSquare: Rect2;
}
export default Rect2;
