"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Abstract2DShape = void 0;
/**
 * An abstract base class for 2D shapes.
 */
class Abstract2DShape {
    /**
     * @returns the distance from `point` to this shape. If `point` is within this shape,
     * this returns the distance from `point` to the edge of this shape.
     *
     * @see {@link signedDistance}
     */
    distance(point) {
        return Math.abs(this.signedDistance(point));
    }
    /**
     * Returns `true` if and only if the given `point` is contained within this shape.
     *
     * `epsilon` is a small number used to counteract floating point error. Thus, if
     * `point` is within `epsilon` of the inside of this shape, `containsPoint` may also
     * return `true`.
     *
     * The default implementation relies on `signedDistance`.
     * Subclasses may override this method to provide a more efficient implementation.
     */
    containsPoint(point, epsilon = Abstract2DShape.smallValue) {
        return this.signedDistance(point) < epsilon;
    }
    /**
     * Returns a bounding box that **loosely** fits the content of this shape.
     *
     * The result of this call can be larger than the result of {@link getTightBoundingBox},
     * **but should not be smaller**. Thus, a call to `getLooseBoundingBox` can be significantly
     * faster than a call to {@link getTightBoundingBox} for some shapes.
     */
    getLooseBoundingBox() {
        return this.getTightBoundingBox();
    }
}
exports.Abstract2DShape = Abstract2DShape;
// @internal
Abstract2DShape.smallValue = 1e-12;
exports.default = Abstract2DShape;
