import Vec3 from './Vec3';
/**
 * Represents a color.
 *
 * @example
 * ```ts,runnable,console
 * import { Color4 } from '@js-draw/math';
 *
 * console.log('Red:', Color4.fromString('#f00'));
 * console.log('Also red:', Color4.ofRGB(1, 0, 0), Color4.red);
 * console.log('Mixing red and blue:', Color4.red.mix(Color4.blue, 0.5));
 * console.log('To string:', Color4.orange.toHexString());
 * ```
 */
export declare class Color4 {
    /** Red component. Should be in the range [0, 1]. */
    readonly r: number;
    /** Green component. ${\tt g} \in [0, 1]$ */
    readonly g: number;
    /** Blue component. ${\tt b} \in [0, 1]$ */
    readonly b: number;
    /** Alpha/transparent component. ${\tt a} \in [0, 1]$. 0 = transparent */
    readonly a: number;
    private constructor();
    /**
     * Create a color from red, green, blue components. The color is fully opaque (`a = 1.0`).
     *
     * Each component should be in the range [0, 1].
     */
    static ofRGB(red: number, green: number, blue: number): Color4;
    /**
     * Creates a color from red, green, blue, and transparency components. Each component should
     * be in the range $[0, 1]$.
     */
    static ofRGBA(red: number, green: number, blue: number, alpha: number): Color4;
    /**
     * Creates a color from an RGB (or RGBA) array.
     *
     * This is similar to {@link ofRGB} and {@link ofRGBA}, but, by default, takes values
     * that range from 0 to 255.
     *
     * If the array values instead range from 0-1, pass `maxValue` as `1`.
     */
    static fromRGBArray(array: Uint8Array | Uint8ClampedArray | number[], maxValue?: number): Color4;
    /**
     * Creates a `Color4` from a three or four-component hexadecimal
     * [color string](https://en.wikipedia.org/wiki/Web_colors#Hex_triplet).
     *
     * Example:
     * ```ts,runnable,console
     * import { Color4 } from '@js-draw/math';
     * console.log(Color4.fromHex('#ff0'));
     * ```
     */
    static fromHex(hexString: string): Color4;
    /** Like {@link fromHex}, but can handle additional colors if an `HTMLCanvasElement` is available. */
    static fromString(text: string): Color4;
    /** @returns true if `this` and `other` are approximately equal. */
    eq(other: Color4 | null | undefined): boolean;
    /**
     * If `fractionTo` is not in the range $[0, 1]$, it will be clamped to the nearest number
     * in that range. For example, `a.mix(b, -1)` is equivalent to `a.mix(b, 0)`.
     *
     * @returns a color `fractionTo` of the way from this color to `other`.
     *
     * @example
     * ```ts
     * Color4.ofRGB(1, 0, 0).mix(Color4.ofRGB(0, 1, 0), 0.1) // -> Color4(0.9, 0.1, 0)
     * ```
     */
    mix(other: Color4, fractionTo: number): Color4;
    /** Returns a new color with a different opacity. */
    withAlpha(a: number): Color4;
    /**
     * Ignoring this color's alpha component, returns a vector with components,
     * $$
     * \begin{pmatrix} \colorbox{#F44}{\tt r} \\ \colorbox{#4F4}{\tt g} \\ \colorbox{#44F}{\tt b} \end{pmatrix}
     * $$
     */
    get rgb(): Vec3;
    /**
     * Returns the [relative luminance](https://www.w3.org/TR/2008/REC-WCAG20-20081211/#relativeluminancedef)
     * of this color in the sRGB color space.
     *
     * Ignores the alpha component.
     */
    relativeLuminance(): number;
    /**
     * Returns the [contrast ratio](https://www.w3.org/TR/2008/REC-WCAG20-20081211/#contrast-ratiodef)
     * between `colorA` and `colorB`.
     */
    static contrastRatio(colorA: Color4, colorB: Color4): number;
    /**
     * @returns the component-wise average of `colors`, or `Color4.transparent` if `colors` is empty.
     */
    static average(colors: Color4[]): Color4;
    /**
     * Converts to (hue, saturation, value).
     * See also https://en.wikipedia.org/wiki/HSL_and_HSV#General_approach
     *
     * The resultant hue is represented in radians and is thus in $[0, 2\pi]$.
     */
    asHSV(): Vec3;
    /**
     * Creates a new `Color4` from a representation [in $HSV$](https://en.wikipedia.org/wiki/HSL_and_HSV#HSV_to_RGB).
     *
     * [Algorithm](https://en.wikipedia.org/wiki/HSL_and_HSV#HSV_to_RGB).
     *
     * Note that hue must be given **in radians**. While non-standard, this is consistent with
     * {@link asHSV}.
     *
     * `hue` and `value` should range from 0 to 1.
     *
     * @param hue $H \in [0, 2\pi]$
     * @param saturation $S_V \in [0, 1]$
     * @param value $V \in [0, 1]$
     */
    static fromHSV(hue: number, saturation: number, value: number): Color4;
    /**
     * Equivalent to `ofRGB(rgb.x, rgb.y, rgb.z)`.
     *
     * All components should be in the range `[0, 1]` (0 to 1 inclusive).
     */
    static fromRGBVector(rgb: Vec3, alpha?: number): Color4;
    private hexString;
    /**
     * @returns a hexadecimal color string representation of `this`, in the form `#rrggbbaa`.
     *
     * @example
     * ```
     * Color4.red.toHexString(); // -> #ff0000ff
     * ```
     */
    toHexString(): string;
    toString(): string;
    static transparent: Color4;
    static red: Color4;
    static orange: Color4;
    static green: Color4;
    static blue: Color4;
    static purple: Color4;
    static yellow: Color4;
    static clay: Color4;
    static black: Color4;
    static gray: Color4;
    static white: Color4;
}
export default Color4;
