"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.__testing__ = exports.removeAccents = exports.calculateTextDocumentOffsets = exports.stringToRegExp = exports.extractText = exports.textOffset = exports.matchCase = exports.camelToSnake = exports.snakeToCamel = exports.lcFirst = exports.ucFirst = exports.isFirstCharacterLower = exports.isFirstCharacterUpper = exports.isLowerCase = exports.isUpperCase = exports.extractWordsFromCodeTextOffset = exports.extractWordsFromCode = exports.extractPossibleWordsFromTextOffset = exports.cleanTextOffset = exports.cleanText = exports.extractWordsFromTextOffset = exports.extractWordsFromText = exports.extractLinesOfText = exports.matchToTextOffset = exports.matchStringToTextOffset = exports.match = exports.splitCamelCaseWord = exports.splitCamelCaseWordWithOffset = void 0;
const cspell_pipe_1 = require("@cspell/cspell-pipe");
const gensequence_1 = require("gensequence");
const search_1 = require("./search");
const textRegex_1 = require("./textRegex");
const util_1 = require("./util");
// CSpell:ignore ings ning gimuy tsmerge
function splitCamelCaseWordWithOffset(wo) {
    return splitCamelCaseWord(wo.text).map((0, util_1.scanMap)((last, text) => ({ text, offset: last.offset + last.text.length }), {
        text: '',
        offset: wo.offset,
    }));
}
exports.splitCamelCaseWordWithOffset = splitCamelCaseWordWithOffset;
/**
 * Split camelCase words into an array of strings.
 */
function splitCamelCaseWord(word) {
    const wPrime = word.replace(textRegex_1.regExUpperSOrIng, (s) => s[0] + s.slice(1).toLowerCase());
    const separator = '_<^*_*^>_';
    const pass1 = wPrime.replace(textRegex_1.regExSplitWords, '$1' + separator + '$2');
    const pass2 = pass1.replace(textRegex_1.regExSplitWords2, '$1' + separator + '$2');
    return pass2.split(separator);
}
exports.splitCamelCaseWord = splitCamelCaseWord;
/**
 * This function lets you iterate over regular expression matches.
 */
function match(reg, text) {
    return (0, gensequence_1.sequenceFromRegExpMatch)(reg, text);
}
exports.match = match;
function matchStringToTextOffset(reg, text) {
    return matchToTextOffset(reg, { text, offset: 0 });
}
exports.matchStringToTextOffset = matchStringToTextOffset;
function matchToTextOffset(reg, text) {
    const textOffset = text;
    const fnOffsetMap = offsetMap(textOffset.offset);
    textOffset.text.matchAll(reg);
    return (0, cspell_pipe_1.pipeSync)(match(reg, textOffset.text), (0, cspell_pipe_1.opMap)((m) => fnOffsetMap({ text: m[0], offset: m.index || 0 })));
}
exports.matchToTextOffset = matchToTextOffset;
function extractLinesOfText(text) {
    return matchStringToTextOffset(textRegex_1.regExLines, text);
}
exports.extractLinesOfText = extractLinesOfText;
/**
 * Extract out whole words from a string of text.
 */
function extractWordsFromText(text) {
    return extractWordsFromTextOffset(textOffset(text));
}
exports.extractWordsFromText = extractWordsFromText;
/**
 * Extract out whole words from a string of text.
 */
function extractWordsFromTextOffset(text) {
    const reg = new RegExp(textRegex_1.regExWords);
    return matchToTextOffset(reg, cleanTextOffset(text));
}
exports.extractWordsFromTextOffset = extractWordsFromTextOffset;
function cleanText(text) {
    text = text.replace(textRegex_1.regExIgnoreCharacters, (match) => ' '.repeat(match.length));
    return text;
}
exports.cleanText = cleanText;
function cleanTextOffset(text) {
    return {
        text: cleanText(text.text),
        offset: text.offset,
    };
}
exports.cleanTextOffset = cleanTextOffset;
/**
 * Extract out whole words and words containing numbers from a string of text.
 */
function extractPossibleWordsFromTextOffset(text) {
    const reg = new RegExp(textRegex_1.regExWordsAndDigits);
    return matchToTextOffset(reg, text);
}
exports.extractPossibleWordsFromTextOffset = extractPossibleWordsFromTextOffset;
function extractWordsFromCode(text) {
    return extractWordsFromCodeTextOffset(textOffset(text));
}
exports.extractWordsFromCode = extractWordsFromCode;
function extractWordsFromCodeTextOffset(textOffset) {
    return (0, cspell_pipe_1.pipeSync)(extractWordsFromTextOffset(textOffset), (0, cspell_pipe_1.opConcatMap)(splitCamelCaseWordWithOffset));
}
exports.extractWordsFromCodeTextOffset = extractWordsFromCodeTextOffset;
function isUpperCase(word) {
    return !!word.match(textRegex_1.regExAllUpper);
}
exports.isUpperCase = isUpperCase;
function isLowerCase(word) {
    return !!word.match(textRegex_1.regExAllLower);
}
exports.isLowerCase = isLowerCase;
function isFirstCharacterUpper(word) {
    return isUpperCase(word.slice(0, 1));
}
exports.isFirstCharacterUpper = isFirstCharacterUpper;
function isFirstCharacterLower(word) {
    return isLowerCase(word.slice(0, 1));
}
exports.isFirstCharacterLower = isFirstCharacterLower;
function ucFirst(word) {
    return word.slice(0, 1).toUpperCase() + word.slice(1);
}
exports.ucFirst = ucFirst;
function lcFirst(word) {
    return word.slice(0, 1).toLowerCase() + word.slice(1);
}
exports.lcFirst = lcFirst;
function snakeToCamel(word) {
    return word.split('_').map(ucFirst).join('');
}
exports.snakeToCamel = snakeToCamel;
function camelToSnake(word) {
    return splitCamelCaseWord(word).join('_').toLowerCase();
}
exports.camelToSnake = camelToSnake;
function matchCase(example, word) {
    if (example.match(textRegex_1.regExFirstUpper)) {
        return word.slice(0, 1).toUpperCase() + word.slice(1).toLowerCase();
    }
    if (example.match(textRegex_1.regExAllLower)) {
        return word.toLowerCase();
    }
    if (example.match(textRegex_1.regExAllUpper)) {
        return word.toUpperCase();
    }
    if (isFirstCharacterUpper(example)) {
        return ucFirst(word);
    }
    if (isFirstCharacterLower(example)) {
        return lcFirst(word);
    }
    return word;
}
exports.matchCase = matchCase;
function textOffset(text, offset = 0) {
    return { text, offset };
}
exports.textOffset = textOffset;
function extractText(textOffset, startPos, endPos) {
    const { text, offset: orig } = textOffset;
    const a = Math.max(startPos - orig, 0);
    const b = Math.max(endPos - orig, 0);
    return text.slice(a, b);
}
exports.extractText = extractText;
function offsetMap(offset) {
    return (xo) => ({ ...xo, offset: xo.offset + offset });
}
function stringToRegExp(pattern, defaultFlags = 'gimu', forceFlags = 'g') {
    if (pattern instanceof RegExp) {
        return pattern;
    }
    try {
        const [, pat, flag] = [...(pattern.match(textRegex_1.regExMatchRegExParts) || ['', pattern, defaultFlags]), forceFlags];
        // Make sure the flags are unique.
        const flags = [...new Set(forceFlags + flag)].join('').replace(/[^gimuy]/g, '');
        if (pat) {
            const regex = new RegExp(pat, flags);
            return regex;
        }
    }
    catch (e) {
        /* empty */
    }
    return undefined;
}
exports.stringToRegExp = stringToRegExp;
function calculateTextDocumentOffsets(uri, doc, wordOffsets) {
    const lines = [
        -1,
        ...(0, cspell_pipe_1.pipeSync)(match(/\n/g, doc), (0, cspell_pipe_1.opMap)((a) => a.index)),
        doc.length,
    ];
    let lastRow = -1;
    let lastOffset = doc.length + 1;
    let lastLineRow = -1;
    let lastLine;
    function findRowCol(offset) {
        const row = (0, search_1.binarySearch)(lines, offset, offset >= lastOffset ? lastRow : undefined);
        const col = offset - lines[Math.max(0, row - 1)];
        lastOffset = offset;
        lastRow = row;
        return [row, col];
    }
    function extractLine(row) {
        const offset = lines[row - 1] + 1;
        const text = doc.slice(offset, lines[row] + 1);
        return { text, offset };
    }
    function calcLine(row) {
        const last = lastLineRow === row ? lastLine : undefined;
        lastLineRow = row;
        const r = last !== null && last !== void 0 ? last : extractLine(row);
        lastLine = r;
        return r;
    }
    return wordOffsets.map((wo) => {
        const [row, col] = findRowCol(wo.offset);
        return { ...wo, row, col, doc, uri, line: calcLine(row) };
    });
}
exports.calculateTextDocumentOffsets = calculateTextDocumentOffsets;
function removeAccents(text) {
    return text.normalize('NFD').replace(textRegex_1.regExAccents, '');
}
exports.removeAccents = removeAccents;
exports.__testing__ = {
    regExWords: textRegex_1.regExWords,
    regExWordsAndDigits: textRegex_1.regExWordsAndDigits,
};
//# sourceMappingURL=text.js.map