"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.createFailedToLoadDictionary = exports.createForbiddenWordsDictionary = exports.createSpellingDictionary = void 0;
const cspell_trie_lib_1 = require("cspell-trie-lib");
const fast_equals_1 = require("fast-equals");
const gensequence_1 = require("gensequence");
const simpleCache_1 = require("../util/simpleCache");
const SpellingDictionaryFromTrie_1 = require("./SpellingDictionaryFromTrie");
const SpellingDictionaryMethods_1 = require("./SpellingDictionaryMethods");
const defaultOptions = Object.freeze({
    weightMap: undefined,
});
const cachedDictionaries = new simpleCache_1.AutoWeakCache(_createSpellingDictionary, 64);
const maxSetSize = 3;
const cachedParamsByWordList = new simpleCache_1.SimpleCache(64);
function createSpellingDictionary(wordList, name, source, options) {
    const params = [wordList, name, source, options];
    if (!Array.isArray(wordList)) {
        return _createSpellingDictionary(params);
    }
    const cached = cachedParamsByWordList.get(name) || new Set();
    for (const cachedParams of cached) {
        if ((0, fast_equals_1.deepEqual)(params, cachedParams)) {
            return cachedDictionaries.get(cachedParams);
        }
    }
    if (cached.size > maxSetSize)
        cached.clear();
    cached.add(params);
    cachedParamsByWordList.set(name, cached);
    return cachedDictionaries.get(params);
}
exports.createSpellingDictionary = createSpellingDictionary;
function _createSpellingDictionary(params) {
    const [wordList, name, source, options] = params;
    // console.log(`createSpellingDictionary ${name} ${source}`);
    const words = (0, cspell_trie_lib_1.parseDictionaryLines)(wordList);
    const trie = (0, cspell_trie_lib_1.buildTrieFast)(words);
    const opts = { ...(options || defaultOptions) };
    if (opts.weightMap === undefined && opts.dictionaryInformation) {
        opts.weightMap = (0, SpellingDictionaryMethods_1.createWeightMapFromDictionaryInformation)(opts.dictionaryInformation);
    }
    return new SpellingDictionaryFromTrie_1.SpellingDictionaryFromTrie(trie, name, opts, source);
}
function createForbiddenWordsDictionary(wordList, name, source, options) {
    // console.log(`createSpellingDictionary ${name} ${source}`);
    const words = (0, cspell_trie_lib_1.parseDictionaryLines)(wordList.concat(wordList.map((a) => a.toLowerCase())), {
        stripCaseAndAccents: !(options === null || options === void 0 ? void 0 : options.noSuggest),
    });
    const forbidWords = gensequence_1.operators.map((w) => '!' + w)(words);
    const trie = (0, cspell_trie_lib_1.buildTrieFast)(forbidWords);
    return new SpellingDictionaryFromTrie_1.SpellingDictionaryFromTrie(trie, name, options || defaultOptions, source);
}
exports.createForbiddenWordsDictionary = createForbiddenWordsDictionary;
function createFailedToLoadDictionary(error) {
    const { options, uri: source } = error;
    const errors = [error];
    return {
        name: options.name,
        source,
        type: 'error',
        containsNoSuggestWords: false,
        has: () => false,
        find: () => undefined,
        isNoSuggestWord: () => false,
        isForbidden: () => false,
        suggest: () => [],
        mapWord: (a) => a,
        genSuggestions: () => {
            return;
        },
        size: 0,
        options,
        isDictionaryCaseSensitive: false,
        getErrors: () => errors,
    };
}
exports.createFailedToLoadDictionary = createFailedToLoadDictionary;
//# sourceMappingURL=createSpellingDictionary.js.map