"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.clearPnPGlobalCache = exports.UnsupportedPnpFile = exports.UnsupportedSchema = exports.pnpLoader = exports.PnpLoader = void 0;
/**
 * Handles loading of `.pnp.js` and `.pnp.js` files.
 */
const vscode_uri_1 = require("vscode-uri");
const import_fresh_1 = __importDefault(require("import-fresh"));
const find_up_1 = __importDefault(require("find-up"));
const clear_module_1 = __importDefault(require("clear-module"));
const defaultPnpFiles = ['.pnp.cjs', '.pnp.js'];
const supportedSchemas = new Set(['file']);
const cachedRequests = new Map();
let lock = undefined;
const cachedPnpImportsSync = new Map();
const cachedRequestsSync = new Map();
class PnpLoader {
    constructor(pnpFiles = defaultPnpFiles) {
        this.pnpFiles = pnpFiles;
        this.cacheKeySuffix = ':' + pnpFiles.join();
    }
    /**
     * Request that the nearest .pnp file gets loaded
     * @param uriDirectory starting directory
     * @returns promise - rejects on error - success if loaded or not found.
     */
    async load(uriDirectory) {
        await lock;
        const cacheKey = this.calcKey(uriDirectory);
        const cached = cachedRequests.get(cacheKey);
        if (cached)
            return cached;
        const r = findPnpAndLoad(uriDirectory, this.pnpFiles);
        cachedRequests.set(cacheKey, r);
        const result = await r;
        cachedRequestsSync.set(cacheKey, result);
        return result;
    }
    async peek(uriDirectory) {
        var _a;
        await lock;
        const cacheKey = this.calcKey(uriDirectory);
        return (_a = cachedRequests.get(cacheKey)) !== null && _a !== void 0 ? _a : Promise.resolve(undefined);
    }
    /**
     * Request that the nearest .pnp file gets loaded
     * @param uriDirectory starting directory
     * @returns promise - rejects on error - success if loaded or not found.
     */
    loadSync(uriDirectory) {
        const cacheKey = this.calcKey(uriDirectory);
        const cached = cachedRequestsSync.get(cacheKey);
        if (cached)
            return cached;
        const r = findPnpAndLoadSync(uriDirectory, this.pnpFiles);
        cachedRequestsSync.set(cacheKey, r);
        cachedRequests.set(cacheKey, Promise.resolve(r));
        return r;
    }
    peekSync(uriDirectory) {
        const cacheKey = this.calcKey(uriDirectory);
        return cachedRequestsSync.get(cacheKey);
    }
    /**
     * Clears the cached so .pnp files will get reloaded on request.
     */
    clearCache() {
        return clearPnPGlobalCache();
    }
    calcKey(uriDirectory) {
        return uriDirectory.toString() + this.cacheKeySuffix;
    }
}
exports.PnpLoader = PnpLoader;
function pnpLoader(pnpFiles) {
    return new PnpLoader(pnpFiles);
}
exports.pnpLoader = pnpLoader;
class UnsupportedSchema extends Error {
    constructor(msg) {
        super(msg);
    }
}
exports.UnsupportedSchema = UnsupportedSchema;
class UnsupportedPnpFile extends Error {
    constructor(msg) {
        super(msg);
    }
}
exports.UnsupportedPnpFile = UnsupportedPnpFile;
/**
 * @param uriDirectory - directory to start at.
 */
async function findPnpAndLoad(uriDirectory, pnpFiles) {
    validateSchema(uriDirectory);
    const found = await (0, find_up_1.default)(pnpFiles, { cwd: uriDirectory.fsPath });
    return loadPnpIfNeeded(found);
}
/**
 * @param uriDirectory - directory to start at.
 */
function findPnpAndLoadSync(uriDirectory, pnpFiles) {
    validateSchema(uriDirectory);
    const found = find_up_1.default.sync(pnpFiles, { cwd: uriDirectory.fsPath });
    return loadPnpIfNeeded(found);
}
function loadPnpIfNeeded(found) {
    if (!found)
        return undefined;
    const c = cachedPnpImportsSync.get(found);
    if (c || cachedPnpImportsSync.has(found))
        return c;
    const r = loadPnp(found);
    cachedPnpImportsSync.set(found, r);
    return r;
}
function loadPnp(pnpFile) {
    const pnp = (0, import_fresh_1.default)(pnpFile);
    if (pnp.setup) {
        pnp.setup();
        return vscode_uri_1.URI.file(pnpFile);
    }
    throw new UnsupportedPnpFile(`Unsupported pnp file: "${pnpFile}"`);
}
function validateSchema(uri) {
    if (!supportedSchemas.has(uri.scheme)) {
        throw new UnsupportedSchema(`Unsupported schema for PNP: "${uri.scheme}"`);
    }
    return true;
}
function clearPnPGlobalCache() {
    if (lock)
        return lock;
    lock = _cleanCache().finally(() => {
        lock = undefined;
    });
    return lock;
}
exports.clearPnPGlobalCache = clearPnPGlobalCache;
async function _cleanCache() {
    await Promise.all([...cachedRequests.values()].map(rejectToUndefined));
    const modules = [...cachedPnpImportsSync.values()];
    modules.forEach((r) => r && clear_module_1.default.single(r.fsPath));
    cachedRequests.clear();
    cachedRequestsSync.clear();
    cachedPnpImportsSync.clear();
    return undefined;
}
function rejectToUndefined(p) {
    return p.catch(() => undefined);
}
//# sourceMappingURL=pnpLoader.js.map