"use strict";
/**
 * The data for this file was constructed from:
 * ```
 * cd ~/projects/clones/vscode/extensions
 * find . -type f -iname package.json -exec pcregrep -M '(?:"id":.*)|(?:"extensions":[^\]]+)' {} \; > ~/projects/cspell/src/languageIds.txt`
 * ```
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getLanguagesForBasename = exports.getLanguagesForExt = exports.buildLanguageExtensionMapSet = exports.isGenerated = exports.isGeneratedFile = exports.isGeneratedExt = exports.isBinary = exports.isBinaryFile = exports.isBinaryExt = exports.languageIds = exports.generatedFiles = exports.binaryLanguages = exports.languageExtensionDefinitions = void 0;
exports.languageExtensionDefinitions = [
    { id: 'ada', extensions: ['.adb', '.ads'] },
    { id: 'apiblueprint', extensions: ['.apib', '.apiblueprint'] },
    { id: 'asciidoc', extensions: ['.adoc', '.asc', '.asciidoc'] },
    { id: 'bat', extensions: ['.bat', '.cmd'] },
    { id: 'clojure', extensions: ['.clj', '.cljs', '.cljx', '.clojure', '.edn'] },
    { id: 'coffeescript', extensions: ['.coffee', '.cson'] },
    { id: 'c', extensions: ['.c'] },
    {
        id: 'cpp',
        extensions: ['.cpp', '.cc', '.cxx', '.hpp', '.hh', '.hxx', '.h', '.mm', '.ino', '.inl'],
    },
    { id: 'csharp', extensions: ['.cs'] },
    { id: 'css', extensions: ['.css'] },
    { id: 'dhall', extensions: ['.dhall'] },
    { id: 'diff', extensions: ['.diff', '.patch', '.rej'] },
    { id: 'dockerfile', extensions: ['.dockerfile'] },
    { id: 'elixir', extensions: ['.ex', '.exs'] },
    { id: 'fsharp', extensions: ['.fs', '.fsi', '.fsx', '.fsscript'] },
    { id: 'go', extensions: ['.go'] },
    { id: 'groovy', extensions: ['.groovy', '.gvy', '.gradle'] },
    { id: 'handlebars', extensions: ['.handlebars', '.hbs'] },
    { id: 'haskell', extensions: ['.hs', '.lhs'] },
    {
        id: 'html',
        extensions: ['.html', '.htm', '.shtml', '.xhtml', '.mdoc', '.jsp', '.asp', '.aspx', '.jshtm', '.volt', '.vue'],
    },
    { id: 'ini', extensions: ['.ini'] },
    { id: 'properties', extensions: ['.properties', '.gitconfig', '.cfg', '.conf'] },
    { id: 'jade', extensions: ['.jade', '.pug'] },
    { id: 'java', extensions: ['.java', '.jav'] },
    { id: 'javascriptreact', extensions: ['.jsx'] },
    { id: 'javascript', extensions: ['.js', '.mjs', '.es6', '.cjs'] },
    {
        id: 'json',
        extensions: ['.json', '.jsonc', '.bowerrc', '.jshintrc', '.jscsrc', '.eslintrc', '.babelrc', '.webmanifest'],
    },
    { id: 'jsonc', extensions: ['.jsonc'] },
    { id: 'jsonc', extensions: [], filenames: ['.code-workspace'] },
    { id: 'jungle', extensions: ['.jungle'] },
    { id: 'less', extensions: ['.less'] },
    { id: 'literate haskell', extensions: ['.lhs'] },
    { id: 'lua', extensions: ['.lua'] },
    { id: 'makefile', extensions: ['.mk'] },
    { id: 'markdown', extensions: ['.md', '.mdown', '.markdown', '.markdn'] },
    { id: 'mdx', extensions: ['.mdx'] },
    { id: 'monkeyc', extensions: ['.mc', '.mb'] },
    { id: 'objective-c', extensions: ['.m'] },
    { id: 'perl', extensions: ['.pl', '.pm', '.pod', '.t', '.PL', '.psgi'] },
    { id: 'perl6', extensions: ['.p6', '.pl6', '.pm6', '.nqp'] },
    { id: 'php', extensions: ['.php', '.php4', '.php5', '.phtml', '.ctp'] },
    { id: 'plaintext', extensions: ['.txt'] },
    { id: 'powershell', extensions: ['.ps1', '.psm1', '.psd1', '.pssc', '.psrc'] },
    { id: 'purescript', extensions: ['.purs'] },
    { id: 'python', extensions: ['.py', '.rpy', '.pyw', '.cpy', '.gyp', '.gypi'] },
    { id: 'r', extensions: ['.r', '.R', '.rhistory', '.rprofile', '.rt'] },
    { id: 'razor', extensions: ['.cshtml'] },
    { id: 'ruby', extensions: ['.rb', '.rbx', '.rjs', '.gemspec', '.rake', '.ru'] },
    { id: 'ruby', extensions: [], filenames: ['Gemfile'] },
    { id: 'rust', extensions: ['.rs'] },
    { id: 'scala', extensions: ['.scala', '.sc'] },
    { id: 'scss', extensions: ['.scss'] },
    { id: 'shaderlab', extensions: ['.shader', '.cginc'] },
    {
        id: 'shellscript',
        extensions: [
            '.sh',
            '.bash',
            '.bashrc',
            '.bash_aliases',
            '.bash_profile',
            '.bash_login',
            '.ebuild',
            '.install',
            '.profile',
            '.bash_logout',
            '.zsh',
            '.zshrc',
            '.zprofile',
            '.zlogin',
            '.zlogout',
            '.zshenv',
            '.zsh-theme',
        ],
    },
    { id: 'sql', extensions: ['.sql', '.dsql'] },
    { id: 'swift', extensions: ['.swift'] },
    { id: 'toml', extensions: ['.toml'] },
    { id: 'typescript', extensions: ['.ts', '.cts', '.mts'] },
    { id: 'typescriptreact', extensions: ['.tsx'] },
    { id: 'vb', extensions: ['.vb', '.brs', '.vbs', '.bas'] },
    { id: 'vue', extensions: ['.vue'] },
    {
        id: 'xml',
        extensions: [
            '.ascx',
            '.atom',
            '.axml',
            '.bpmn',
            '.config',
            '.cpt',
            '.csl',
            '.csproj.user',
            '.csproj',
            '.dita',
            '.ditamap',
            '.dtd',
            '.dtml',
            '.ent',
            '.fsproj',
            '.fxml',
            '.iml',
            '.isml',
            '.jmx',
            '.launch',
            '.menu',
            '.mod',
            '.mxml',
            '.nuspec',
            '.opml',
            '.owl',
            '.proj',
            '.pt',
            '.pubxml.user',
            '.pubxml',
            '.rdf',
            '.rng',
            '.rss',
            '.shproj',
            '.storyboard',
            '.svg',
            '.targets',
            '.tld',
            '.tmx',
            '.vbproj.user',
            '.vbproj',
            '.vcxproj.filters',
            '.vcxproj',
            '.wsdl',
            '.wxi',
            '.wxl',
            '.wxs',
            '.xaml',
            '.xbl',
            '.xib',
            '.xlf',
            '.xliff',
            '.xml',
            '.xoml',
            '.xpdl',
            '.xsd',
            '.xul',
        ],
    },
    { id: 'xsl', extensions: ['.xsl', '.xslt'] },
    { id: 'yaml', extensions: ['.eyaml', '.eyml', '.yaml', '.yml'] },
    { id: 'latex', extensions: ['.tex'] },
    { id: 'map', extensions: ['.map'] },
    { id: 'pdf', extensions: ['.pdf'] },
    { id: 'rsa', extensions: ['.pub'], filenames: ['id_rsa', 'id_rsa.pub'] },
    { id: 'pem', extensions: ['.private-key.pem', '.pem'] },
    { id: 'pem-private-key', extensions: ['.private-key.pem'] },
    //
    // Special file types used to prevent spell checking.
    //
    { id: 'image', extensions: ['.jpg', '.png', '.jpeg', '.tiff', '.bmp', '.gif', '.ico'] },
    // cspell:ignore woff
    {
        id: 'binary',
        extensions: ['.gz', '.exe', '.dll', '.lib', '.obj', '.o', '.eot', '.cur', '.zip'],
    },
    {
        id: 'fonts',
        extensions: ['.ttf', '.woff', '.woff2'],
    },
    {
        id: 'video',
        extensions: ['.mov', '.mpg'],
    },
    {
        id: 'lock',
        extensions: ['.lock'],
        filenames: ['package-lock.json'],
    },
    {
        id: 'cache_files',
        extensions: [],
        // cspell:ignore eslintcache
        filenames: ['.cspellcache', '.DS_Store', '.eslintcache'],
    },
];
exports.binaryLanguages = new Set(['binary', 'image', 'video', 'fonts']);
exports.generatedFiles = new Set([...exports.binaryLanguages, 'map', 'lock', 'pdf', 'cache_files', 'rsa', 'pem']);
exports.languageIds = exports.languageExtensionDefinitions.map(({ id }) => id);
const mapExtensionToSetOfLanguageIds = buildLanguageExtensionMapSet(exports.languageExtensionDefinitions);
const mapExtensionToLanguageIds = buildExtensionToLanguageIdMap(mapExtensionToSetOfLanguageIds);
function isBinaryExt(ext) {
    return isBinary(getLanguagesForExt(ext));
}
exports.isBinaryExt = isBinaryExt;
function isBinaryFile(basename) {
    return isBinary(getLanguagesForBasename(basename));
}
exports.isBinaryFile = isBinaryFile;
function isBinary(languageId) {
    return doesSetContainAnyOf(exports.binaryLanguages, languageId);
}
exports.isBinary = isBinary;
function isGeneratedExt(ext) {
    return isGenerated(getLanguagesForExt(ext));
}
exports.isGeneratedExt = isGeneratedExt;
function isGeneratedFile(basename) {
    return isGenerated(getLanguagesForBasename(basename));
}
exports.isGeneratedFile = isGeneratedFile;
function isGenerated(languageId) {
    return doesSetContainAnyOf(exports.generatedFiles, languageId);
}
exports.isGenerated = isGenerated;
function doesSetContainAnyOf(setOfIds, languageId) {
    if (typeof languageId === 'string') {
        return setOfIds.has(languageId);
    }
    for (const id of languageId) {
        if (setOfIds.has(id)) {
            return true;
        }
    }
    return false;
}
function buildLanguageExtensionMapSet(defs) {
    return defs.reduce((map, def) => {
        var _a;
        function getMapSet(value) {
            const found = map.get(value);
            if (found)
                return found;
            const s = new Set();
            map.set(value, s);
            return s;
        }
        function addId(value) {
            getMapSet(value).add(def.id);
        }
        def.extensions.forEach(addId);
        (_a = def.filenames) === null || _a === void 0 ? void 0 : _a.forEach(addId);
        return map;
    }, new Map());
}
exports.buildLanguageExtensionMapSet = buildLanguageExtensionMapSet;
function buildExtensionToLanguageIdMap(map) {
    return new Map([...map].map(([k, s]) => [k, [...s]]));
}
function getLanguagesForExt(ext) {
    return mapExtensionToLanguageIds.get(ext) || mapExtensionToLanguageIds.get('.' + ext) || [];
}
exports.getLanguagesForExt = getLanguagesForExt;
function getLanguagesForBasename(basename) {
    const found = mapExtensionToLanguageIds.get(basename);
    if (found)
        return found;
    for (let pos = basename.indexOf('.'); pos >= 0; pos = basename.indexOf('.', pos + 1)) {
        const ids = mapExtensionToLanguageIds.get(basename.slice(pos));
        if (ids)
            return ids;
    }
    return [];
}
exports.getLanguagesForBasename = getLanguagesForBasename;
//# sourceMappingURL=LanguageIds.js.map