import { SelectionMode, } from 'js-draw';
import  README  from './icons/README.mjs';
import  ExpandMore  from './icons/ExpandMore.mjs';
import  Undo  from './icons/Undo.mjs';
import  Redo  from './icons/Redo.mjs';
import  InkEraser  from './icons/InkEraser.mjs';
import  InkEraserOff  from './icons/InkEraserOff.mjs';
import  PanTool  from './icons/PanTool.mjs';
import  TouchApp  from './icons/TouchApp.mjs';
import  ScreenLockRotation  from './icons/ScreenLockRotation.mjs';
import  Imagesmode  from './icons/Imagesmode.mjs';
import  Title  from './icons/Title.mjs';
import  Resize  from './icons/Resize.mjs';
import  Delete  from './icons/Delete.mjs';
import  ContentCopy  from './icons/ContentCopy.mjs';
import  EditDocument  from './icons/EditDocument.mjs';
import  Check  from './icons/Check.mjs';
import  InkHighlighter  from './icons/InkHighlighter.mjs';
import  Edit  from './icons/Edit.mjs';
import  CloudUpload  from './icons/CloudUpload.mjs';
import  RotateLeft  from './icons/RotateLeft.mjs';
import  Select  from './icons/Select.mjs';
import  Close  from './icons/Close.mjs';
import  Shapes  from './icons/Shapes.mjs';
import  Draw  from './icons/Draw.mjs';
import  InkPen  from './icons/InkPen.mjs';
import  ContentPaste  from './icons/ContentPaste.mjs';
import  LassoSelect  from './icons/LassoSelect.mjs';
import  LineWeight  from './icons/LineWeight.mjs';
/**
 * Converts an icon to an HTML element.
 *
 * This function accepts an "opaque" type to avoid unsafely creating icon DOM elements
 * from untrusted strings.
 */
const icon = (data) => {
    const icon = document.createElement('div');
    // eslint-disable-next-line no-unsanitized/property -- data must not be user-provided (enforced with a custom type).
    icon.innerHTML = data;
    return icon.childNodes[0];
};
/**
 * Creates a `MaterialIconProvider` class bound to a particular instance of the `js-draw`
 * library.
 *
 * **Example**:
 * ```ts,runnable
 * import * as jsdraw from 'js-draw';
 * import { makeMaterialIconProviderClass } from '@js-draw/material-icons';
 * const MaterialIconProvider = makeMaterialIconProviderClass(jsdraw);
 *
 * (new jsdraw.Editor(
 *     document.body, { iconProvider: new MaterialIconProvider() },
 * )).addToolbar();
 * ```
 *
 * If you only have one instance of `js-draw` (which is almost always the case), prefer
 * `new` {@link MaterialIconProvider}.
 *
 * @returns an uninstantiated subclass of {@link js-draw!IconProvider | IconProvider}.
 */
const makeMaterialIconProviderClass = ({ IconProvider, EraserMode, }) => class extends IconProvider {
    makeUndoIcon() {
        return icon(Undo);
    }
    makeRedoIcon() {
        return icon(Redo);
    }
    makeDropdownIcon() {
        return icon(ExpandMore);
    }
    makeEraserIcon(_eraserSize, mode) {
        return icon(mode === EraserMode.PartialStroke ? InkEraserOff : InkEraser);
    }
    makeSelectionIcon(mode) {
        return icon(mode === SelectionMode.Lasso ? LassoSelect : Select);
    }
    makeRotateIcon() {
        return icon(RotateLeft);
    }
    makeHandToolIcon() {
        return icon(PanTool);
    }
    makeTouchPanningIcon() {
        return icon(TouchApp);
    }
    // makeAllDevicePanningIcon(): IconElemType;
    // makeZoomIcon(): IconElemType;
    makeRotationLockIcon() {
        return icon(ScreenLockRotation);
    }
    makeInsertImageIcon() {
        return icon(Imagesmode);
    }
    makeUploadFileIcon() {
        return icon(CloudUpload);
    }
    makeTextIcon(_textStyle) {
        return icon(Title);
    }
    makePenIcon(style) {
        let baseIcon = this.isRoundedTipPen(style) ? Edit : InkHighlighter;
        if (this.isPolylinePen(style)) {
            baseIcon = InkPen;
        }
        const svg = icon(baseIcon);
        svg.setAttribute('viewBox', '0 -880 960 1000');
        const line = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        line.setAttribute('d', `
			M110,-25 L850,-25
		`);
        line.style.stroke = style.color.toHexString();
        line.style.strokeWidth = `${Math.sqrt(style.thickness) * 20}`;
        if (!this.isRoundedTipPen(style)) {
            line.style.strokeLinecap = 'square';
        }
        else {
            line.style.strokeLinecap = 'round';
        }
        svg.insertAdjacentElement('afterbegin', line);
        // Add a grid background to make transparency visible
        if (style.color.a < 1) {
            const checkerboard = this.makeCheckerboardPattern();
            const defs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
            defs.appendChild(checkerboard.patternDefElement);
            svg.appendChild(defs);
            const lineBackground = line.cloneNode();
            lineBackground.style.stroke = checkerboard.patternRef;
            svg.insertAdjacentElement('afterbegin', lineBackground);
        }
        return svg;
    }
    // makeIconFromFactory(pen: Pen, factory: ComponentBuilderFactory, includeTransparencyGrid?: boolean): IconElemType;
    // makePipetteIcon(color?: Color4): IconElemType;
    // makeFormatSelectionIcon(): IconElemType;
    makeShapeAutocorrectIcon() {
        return icon(Shapes);
    }
    makeStrokeSmoothingIcon() {
        return icon(Draw);
    }
    makePressureSensitivityIcon() {
        return icon(LineWeight);
    }
    makeResizeImageToSelectionIcon() {
        return icon(Resize);
    }
    makeDuplicateSelectionIcon() {
        return icon(ContentCopy);
    }
    makeCopyIcon() {
        return icon(ContentCopy);
    }
    makePasteIcon() {
        return icon(ContentPaste);
    }
    makeDeleteSelectionIcon() {
        return icon(Delete);
    }
    makeCloseIcon() {
        return icon(Close);
    }
    makeSaveIcon() {
        return icon(Check);
    }
    makeConfigureDocumentIcon() {
        return icon(EditDocument);
    }
    // makeOverflowIcon(): IconElemType;
    licenseInfo() {
        return README;
    }
};
export default makeMaterialIconProviderClass;
