"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const js_draw_1 = require("js-draw");
const README_1 = __importDefault(require("./icons/README"));
const ExpandMore_1 = __importDefault(require("./icons/ExpandMore"));
const Undo_1 = __importDefault(require("./icons/Undo"));
const Redo_1 = __importDefault(require("./icons/Redo"));
const InkEraser_1 = __importDefault(require("./icons/InkEraser"));
const InkEraserOff_1 = __importDefault(require("./icons/InkEraserOff"));
const PanTool_1 = __importDefault(require("./icons/PanTool"));
const TouchApp_1 = __importDefault(require("./icons/TouchApp"));
const ScreenLockRotation_1 = __importDefault(require("./icons/ScreenLockRotation"));
const Imagesmode_1 = __importDefault(require("./icons/Imagesmode"));
const Title_1 = __importDefault(require("./icons/Title"));
const Resize_1 = __importDefault(require("./icons/Resize"));
const Delete_1 = __importDefault(require("./icons/Delete"));
const ContentCopy_1 = __importDefault(require("./icons/ContentCopy"));
const EditDocument_1 = __importDefault(require("./icons/EditDocument"));
const Check_1 = __importDefault(require("./icons/Check"));
const InkHighlighter_1 = __importDefault(require("./icons/InkHighlighter"));
const Edit_1 = __importDefault(require("./icons/Edit"));
const CloudUpload_1 = __importDefault(require("./icons/CloudUpload"));
const RotateLeft_1 = __importDefault(require("./icons/RotateLeft"));
const Select_1 = __importDefault(require("./icons/Select"));
const Close_1 = __importDefault(require("./icons/Close"));
const Shapes_1 = __importDefault(require("./icons/Shapes"));
const Draw_1 = __importDefault(require("./icons/Draw"));
const InkPen_1 = __importDefault(require("./icons/InkPen"));
const ContentPaste_1 = __importDefault(require("./icons/ContentPaste"));
const LassoSelect_1 = __importDefault(require("./icons/LassoSelect"));
const LineWeight_1 = __importDefault(require("./icons/LineWeight"));
/**
 * Converts an icon to an HTML element.
 *
 * This function accepts an "opaque" type to avoid unsafely creating icon DOM elements
 * from untrusted strings.
 */
const icon = (data) => {
    const icon = document.createElement('div');
    // eslint-disable-next-line no-unsanitized/property -- data must not be user-provided (enforced with a custom type).
    icon.innerHTML = data;
    return icon.childNodes[0];
};
/**
 * Creates a `MaterialIconProvider` class bound to a particular instance of the `js-draw`
 * library.
 *
 * **Example**:
 * ```ts,runnable
 * import * as jsdraw from 'js-draw';
 * import { makeMaterialIconProviderClass } from '@js-draw/material-icons';
 * const MaterialIconProvider = makeMaterialIconProviderClass(jsdraw);
 *
 * (new jsdraw.Editor(
 *     document.body, { iconProvider: new MaterialIconProvider() },
 * )).addToolbar();
 * ```
 *
 * If you only have one instance of `js-draw` (which is almost always the case), prefer
 * `new` {@link MaterialIconProvider}.
 *
 * @returns an uninstantiated subclass of {@link js-draw!IconProvider | IconProvider}.
 */
const makeMaterialIconProviderClass = ({ IconProvider, EraserMode, }) => class extends IconProvider {
    makeUndoIcon() {
        return icon(Undo_1.default);
    }
    makeRedoIcon() {
        return icon(Redo_1.default);
    }
    makeDropdownIcon() {
        return icon(ExpandMore_1.default);
    }
    makeEraserIcon(_eraserSize, mode) {
        return icon(mode === EraserMode.PartialStroke ? InkEraserOff_1.default : InkEraser_1.default);
    }
    makeSelectionIcon(mode) {
        return icon(mode === js_draw_1.SelectionMode.Lasso ? LassoSelect_1.default : Select_1.default);
    }
    makeRotateIcon() {
        return icon(RotateLeft_1.default);
    }
    makeHandToolIcon() {
        return icon(PanTool_1.default);
    }
    makeTouchPanningIcon() {
        return icon(TouchApp_1.default);
    }
    // makeAllDevicePanningIcon(): IconElemType;
    // makeZoomIcon(): IconElemType;
    makeRotationLockIcon() {
        return icon(ScreenLockRotation_1.default);
    }
    makeInsertImageIcon() {
        return icon(Imagesmode_1.default);
    }
    makeUploadFileIcon() {
        return icon(CloudUpload_1.default);
    }
    makeTextIcon(_textStyle) {
        return icon(Title_1.default);
    }
    makePenIcon(style) {
        let baseIcon = this.isRoundedTipPen(style) ? Edit_1.default : InkHighlighter_1.default;
        if (this.isPolylinePen(style)) {
            baseIcon = InkPen_1.default;
        }
        const svg = icon(baseIcon);
        svg.setAttribute('viewBox', '0 -880 960 1000');
        const line = document.createElementNS('http://www.w3.org/2000/svg', 'path');
        line.setAttribute('d', `
			M110,-25 L850,-25
		`);
        line.style.stroke = style.color.toHexString();
        line.style.strokeWidth = `${Math.sqrt(style.thickness) * 20}`;
        if (!this.isRoundedTipPen(style)) {
            line.style.strokeLinecap = 'square';
        }
        else {
            line.style.strokeLinecap = 'round';
        }
        svg.insertAdjacentElement('afterbegin', line);
        // Add a grid background to make transparency visible
        if (style.color.a < 1) {
            const checkerboard = this.makeCheckerboardPattern();
            const defs = document.createElementNS('http://www.w3.org/2000/svg', 'defs');
            defs.appendChild(checkerboard.patternDefElement);
            svg.appendChild(defs);
            const lineBackground = line.cloneNode();
            lineBackground.style.stroke = checkerboard.patternRef;
            svg.insertAdjacentElement('afterbegin', lineBackground);
        }
        return svg;
    }
    // makeIconFromFactory(pen: Pen, factory: ComponentBuilderFactory, includeTransparencyGrid?: boolean): IconElemType;
    // makePipetteIcon(color?: Color4): IconElemType;
    // makeFormatSelectionIcon(): IconElemType;
    makeShapeAutocorrectIcon() {
        return icon(Shapes_1.default);
    }
    makeStrokeSmoothingIcon() {
        return icon(Draw_1.default);
    }
    makePressureSensitivityIcon() {
        return icon(LineWeight_1.default);
    }
    makeResizeImageToSelectionIcon() {
        return icon(Resize_1.default);
    }
    makeDuplicateSelectionIcon() {
        return icon(ContentCopy_1.default);
    }
    makeCopyIcon() {
        return icon(ContentCopy_1.default);
    }
    makePasteIcon() {
        return icon(ContentPaste_1.default);
    }
    makeDeleteSelectionIcon() {
        return icon(Delete_1.default);
    }
    makeCloseIcon() {
        return icon(Close_1.default);
    }
    makeSaveIcon() {
        return icon(Check_1.default);
    }
    makeConfigureDocumentIcon() {
        return icon(EditDocument_1.default);
    }
    // makeOverflowIcon(): IconElemType;
    licenseInfo() {
        return README_1.default;
    }
};
exports.default = makeMaterialIconProviderClass;
