import Editor from '../Editor';
/**
 * Adjusts the current editor theme such that colors have appropriate contrast.
 *
 * As this method overrides CSS variables using the `.style` property,
 * **assumes** all original theme variables are set using CSS and not the `.style` property.
 *
 * If the editor changes theme in response to the system theme, this method should be
 * called whenever the system theme changes (e.g. by using [the `matchMedia`](https://developer.mozilla.org/en-US/docs/Web/API/Window/matchMedia)
 * method).
 *
 * @example
 * ```ts,runnable
 * import { Editor, adjustEditorThemeForContrast } from 'js-draw';
 *
 * const editor = new Editor(document.body);
 * editor.addToolbar();
 *
 * const css = `
 *   :root .imageEditorContainer {
 *     --background-color-1: #ffff77;
 *     --foreground-color-1: #fff;
 *     --background-color-2: #ffff99;
 *     --foreground-color-2: #ffff88;
 *     --background-color-3: #ddffff;
 *     --foreground-color-3: #eeffff;
 *     --selection-background-color: #9f7;
 *     --selection-foreground-color: #98f;
 *   }
 *
 *   @media screen and (prefers-color-scheme: dark) {
 *     :root .imageEditorContainer {
 *       --background-color-1: black;
 *     }
 *   }
 * `;
 * editor.addStyleSheet(css);
 *
 * adjustEditorThemeForContrast(editor);
 *
 * // Because adjustEditorThemeForContrast overrides the current theme, it should be called again
 * // to allow the editor to switch between light/dark themes.
 * window.matchMedia('(prefers-color-scheme: dark)').addEventListener('change', () => {
 *   adjustEditorThemeForContrast(editor);
 * });
 *
 * window.matchMedia('print').addEventListener('change', () => {
 *   adjustEditorThemeForContrast(editor);
 * });
 * ```
 */
declare const adjustEditorThemeForContrast: (editor: Editor, options?: {
    dontClearOverrides: boolean;
}) => void;
export default adjustEditorThemeForContrast;
