import  { assertUnreachable }  from '../../util/assertions.mjs';
import { Rect2, Vec2 } from '@js-draw/math';
import  { cssPrefix }  from './SelectionTool.mjs';
var HandleShape;
(function (HandleShape) {
    HandleShape[HandleShape["Circle"] = 0] = "Circle";
    HandleShape[HandleShape["Square"] = 1] = "Square";
})(HandleShape || (HandleShape = {}));
export var HandleAction;
(function (HandleAction) {
    HandleAction["ResizeXY"] = "resize-xy";
    HandleAction["Rotate"] = "rotate";
    HandleAction["ResizeX"] = "resize-x";
    HandleAction["ResizeY"] = "resize-y";
})(HandleAction || (HandleAction = {}));
// The *interactable* handle size. The visual size will be slightly smaller.
export const handleSize = 30;
export default class SelectionHandle {
    constructor(presentation, parent, viewport, onDragStart, onDragUpdate, onDragEnd) {
        this.presentation = presentation;
        this.parent = parent;
        this.viewport = viewport;
        this.onDragStart = onDragStart;
        this.onDragUpdate = onDragUpdate;
        this.onDragEnd = onDragEnd;
        this.dragLastPos = null;
        this.element = document.createElement('div');
        this.element.classList.add(`${cssPrefix}handle`, `${cssPrefix}${presentation.action}`);
        // Create a slightly smaller content/background element.
        const visibleContent = document.createElement('div');
        visibleContent.classList.add(`${cssPrefix}content`);
        this.element.appendChild(visibleContent);
        this.parentSide = presentation.side;
        const icon = presentation.icon;
        if (icon) {
            visibleContent.appendChild(icon);
            icon.classList.add('icon');
        }
        if (presentation.action === HandleAction.Rotate) {
            this.shape = HandleShape.Circle;
        }
        else {
            this.shape = HandleShape.Square;
        }
        switch (this.shape) {
            case HandleShape.Circle:
                this.element.classList.add(`${cssPrefix}circle`);
                break;
            case HandleShape.Square:
                this.element.classList.add(`${cssPrefix}square`);
                break;
            default:
                assertUnreachable(this.shape);
        }
        this.updatePosition();
    }
    /**
     * Adds this to `container`, where `conatiner` should be the background/selection
     * element visible on the screen.
     */
    addTo(container) {
        container.appendChild(this.element);
    }
    /**
     * Removes this element from its container. Should only be called
     * after {@link addTo}.
     */
    remove() {
        this.element.remove();
    }
    /**
     * Returns this handle's bounding box relative to the top left of the
     * selection box.
     */
    getBBoxParentCoords() {
        const parentRect = this.parent.getScreenRegion();
        const size = Vec2.of(handleSize, handleSize);
        const topLeft = parentRect.size
            .scale(this.parentSide)
            // Center
            .minus(size.times(1 / 2));
        return new Rect2(topLeft.x, topLeft.y, size.x, size.y);
    }
    /** @returns this handle's bounding box relative to the canvas. */
    getBBoxCanvasCoords() {
        const parentRect = this.parent.region;
        const size = Vec2.of(handleSize, handleSize).times(1 / this.viewport.getScaleFactor());
        const topLeftFromParent = parentRect.size.scale(this.parentSide).minus(size.times(0.5));
        return new Rect2(topLeftFromParent.x, topLeftFromParent.y, size.x, size.y).translatedBy(parentRect.topLeft);
    }
    /**
     * Moves the HTML representation of this to the location matching its internal representation.
     */
    updatePosition() {
        const bbox = this.getBBoxParentCoords();
        // Position within the selection box.
        this.element.style.marginLeft = `${bbox.topLeft.x}px`;
        this.element.style.marginTop = `${bbox.topLeft.y}px`;
        this.element.style.width = `${bbox.w}px`;
        this.element.style.height = `${bbox.h}px`;
    }
    /** @returns true iff `point` (in editor **canvas** coordinates) is in this. */
    containsPoint(point) {
        const bbox = this.getBBoxCanvasCoords();
        const delta = point.minus(bbox.center);
        // Should have same x and y radius
        const radius = bbox.size.x / 2;
        let result;
        if (this.shape === HandleShape.Circle) {
            result = delta.magnitude() <= radius;
        }
        else {
            result = Math.abs(delta.x) <= radius && Math.abs(delta.y) <= radius;
        }
        return result;
    }
    handleDragStart(pointer) {
        this.onDragStart(pointer.canvasPos);
        this.dragLastPos = pointer.canvasPos;
        return true;
    }
    handleDragUpdate(pointer) {
        if (!this.dragLastPos) {
            return;
        }
        this.onDragUpdate(pointer.canvasPos);
    }
    handleDragEnd() {
        if (!this.dragLastPos) {
            return;
        }
        return this.onDragEnd();
    }
    setSnapToGrid(snap) {
        this.snapToGrid = snap;
    }
    isSnappingToGrid() {
        return this.snapToGrid;
    }
}
