import { Editor } from '../Editor';
import { Point2 } from '@js-draw/math';
import Pointer from '../Pointer';
import { KeyPressEvent, PointerEvt, WheelEvt } from '../inputEvents';
import BaseTool from './BaseTool';
interface PinchData {
    canvasCenter: Point2;
    screenCenter: Point2;
    angle: number;
    dist: number;
}
export declare enum PanZoomMode {
    /** Touch gestures with a single pointer. Ignores non-touch gestures. */
    OneFingerTouchGestures = 1,
    /** Touch gestures with exactly two pointers. Ignores non-touch gestures. */
    TwoFingerTouchGestures = 2,
    RightClickDrags = 4,
    /** Single-pointer gestures of *any* type (including touch). */
    SinglePointerGestures = 8,
    /** Keyboard navigation (e.g. LeftArrow to move left). */
    Keyboard = 16,
    /** If provided, prevents **this** tool from rotating the viewport (other tools may still do so). */
    RotationLocked = 32
}
/**
 * This tool moves the viewport in response to touchpad, touchscreen, mouse, and keyboard events.
 *
 * Which events are handled, and which are skipped, are determined by the tool's `mode`. For example,
 * a `PanZoom` tool with `mode = PanZoomMode.TwoFingerTouchGestures|PanZoomMode.RightClickDrags` would
 * respond to right-click drag events and two-finger touch gestures.
 *
 * @see {@link setModeEnabled}
 */
export default class PanZoom extends BaseTool {
    private editor;
    private mode;
    private transform;
    private readonly initialRotationSnapAngle;
    private readonly afterRotationStartSnapAngle;
    private readonly pinchZoomStartThreshold;
    private startTouchDist;
    private lastTouchDist;
    private lastScreenCenter;
    private lastTimestamp;
    private lastPointerDownTimestamp;
    private initialTouchAngle;
    private initialViewportRotation;
    private initialViewportScale;
    private isScaling;
    private isRotating;
    private inertialScroller;
    private velocity;
    constructor(editor: Editor, mode: PanZoomMode, description: string);
    canReceiveInputInReadOnlyEditor(): boolean;
    computePinchData(p1: Pointer, p2: Pointer): PinchData;
    private allPointersAreOfType;
    onPointerDown({ allPointers: pointers, current: currentPointer, }: PointerEvt): boolean;
    private updateVelocity;
    private getCenterDelta;
    private toSnappedRotationDelta;
    /**
     * Given a scale update, `scaleFactor`, returns a new scale factor snapped
     * to a power of two (if within some tolerance of that scale).
     */
    private toSnappedScaleFactor;
    private handleTwoFingerMove;
    private handleOneFingerMove;
    onPointerMove({ allPointers }: PointerEvt): void;
    onPointerUp(event: PointerEvt): void;
    onGestureCancel(): void;
    private updateTransform;
    /**
     * Updates the current transform and clears it. Use this method for events that are not part of
     * a larger gesture (i.e. have no start and end event). For example, this would be used for `onwheel`
     * events, but not for `onpointer` events.
     */
    private applyAndFinalizeTransform;
    onWheel({ delta, screenPos }: WheelEvt): boolean;
    onKeyPress(event: KeyPressEvent): boolean;
    private isRotationLocked;
    /**
     * Changes the types of gestures used by this pan/zoom tool.
     *
     * @see {@link PanZoomMode} {@link setMode}
     *
     * @example
     * ```ts,runnable
     * import { Editor, PanZoomTool, PanZoomMode } from 'js-draw';
     *
     * const editor = new Editor(document.body);
     *
     * // By default, there are multiple PanZoom tools that handle different events.
     * // This gets all PanZoomTools.
     * const panZoomToolList = editor.toolController.getMatchingTools(PanZoomTool);
     *
     * // The first PanZoomTool is the highest priority -- by default,
     * // this tool is responsible for handling multi-finger touch gestures.
     * //
     * // Lower-priority PanZoomTools handle one-finger touch gestures and
     * // key-presses.
     * const panZoomTool = panZoomToolList[0];
     *
     * // Lock rotation for multi-finger touch gestures.
     * panZoomTool.setModeEnabled(PanZoomMode.RotationLocked, true);
     * ```
     */
    setModeEnabled(mode: PanZoomMode, enabled: boolean): void;
    /**
     * Sets all modes for this tool using a bitmask.
     *
     * @see {@link setModeEnabled}
     *
     * @example
     * ```ts
     * tool.setMode(PanZoomMode.RotationLocked|PanZoomMode.TwoFingerTouchGestures);
     * ```
     */
    setMode(mode: PanZoomMode): void;
    /**
     * Returns a bitmask indicating the currently-enabled modes.
     * @see {@link setModeEnabled}
     */
    getMode(): PanZoomMode;
}
export {};
