/** Handles filtering and other operations on an image. */
export class ImageWrapper {
    constructor(imageBase64Url, preview, onUrlUpdate) {
        this.imageBase64Url = imageBase64Url;
        this.preview = preview;
        this.onUrlUpdate = onUrlUpdate;
        this.originalSrc = imageBase64Url;
        preview.src = imageBase64Url;
    }
    updateImageData(base64DataUrl) {
        this.preview.src = base64DataUrl;
        this.imageBase64Url = base64DataUrl;
        this.onUrlUpdate();
    }
    decreaseSize(resizeFactor = 3 / 4) {
        const canvas = document.createElement('canvas');
        canvas.width = this.preview.naturalWidth * resizeFactor;
        canvas.height = this.preview.naturalHeight * resizeFactor;
        const ctx = canvas.getContext('2d');
        ctx?.drawImage(this.preview, 0, 0, canvas.width, canvas.height);
        // JPEG can be much smaller than PNG for the same image size. Prefer it if
        // the image is already a JPEG.
        const format = this.originalSrc?.startsWith('data:image/jpeg;') ? 'image/jpeg' : 'image/png';
        this.updateImageData(canvas.toDataURL(format));
    }
    reset() {
        this.updateImageData(this.originalSrc);
    }
    isChanged() {
        return this.imageBase64Url !== this.originalSrc;
    }
    // Returns true if the current image is large enough to display a "decrease size"
    // option.
    isLarge() {
        const largeImageThreshold = 0.12 * 1024 * 1024; // 0.12 MiB
        return this.getBase64Url().length > largeImageThreshold;
    }
    getBase64Url() {
        return this.imageBase64Url;
    }
    getAltText() {
        return this.altText;
    }
    setAltText(text) {
        this.altText = text;
        this.preview.alt = text;
    }
    static fromSrcAndPreview(initialBase64Src, preview, onUrlUpdate) {
        return new ImageWrapper(initialBase64Src, preview, onUrlUpdate);
    }
    static fromRenderable(renderable, onUrlUpdate) {
        const preview = new Image();
        preview.src = renderable.base64Url;
        const result = new ImageWrapper(renderable.base64Url, preview, onUrlUpdate);
        const altText = renderable.label ?? renderable.image.getAttribute('alt');
        if (altText) {
            result.setAltText(altText);
        }
        return { wrapper: result, preview };
    }
}
