import Editor from '../../Editor';
import { KeyPressEvent } from '../../inputEvents';
import { ToolbarLocalization } from '../localization';
import { WidgetContentLayoutManager } from './layout/types';
import HelpDisplay from '../utils/HelpDisplay';
export type SavedToolbuttonState = Record<string, any>;
/**
 * A set of labels that allow toolbar themes to treat buttons differently.
 */
export declare enum ToolbarWidgetTag {
    Save = "save",
    Exit = "exit",
    Undo = "undo",
    Redo = "redo"
}
/**
 * The `abstract` base class for items that can be shown in a `js-draw` toolbar. See also {@link AbstractToolbar.addWidget}.
 *
 * See [the custom tool example](https://github.com/personalizedrefrigerator/js-draw/blob/main/docs/examples/example-custom-tools/example.ts)
 * for how to create a custom toolbar widget for a tool.
 *
 * For custom action buttons, {@link AbstractToolbar.addActionButton} may be sufficient for most use cases.
 */
export default abstract class BaseWidget {
    #private;
    protected editor: Editor;
    protected id: string;
    protected readonly container: HTMLElement;
    private button;
    private icon;
    private layoutManager;
    private dropdown;
    private dropdownContent;
    private dropdownIcon;
    private label;
    private disabled;
    private subWidgets;
    private toplevel;
    protected readonly localizationTable: ToolbarLocalization;
    constructor(editor: Editor, id: string, localizationTable?: ToolbarLocalization);
    /**
     * Should return a constant true or false value. If true (the default),
     * this widget must be automatically disabled when its editor is read-only.
     */
    protected shouldAutoDisableInReadOnlyEditor(): boolean;
    getId(): string;
    /**
     * Note: Tags should be set *before* a tool widget is added to a toolbar.
     *
     *
     * Associates tags with this widget that can be used by toolbar themes
     * to customize the layout/appearance of this button. Prefer tags in
     * the `ToolbarWidgetTag` enum, where possible.
     *
     * In addition to being readable from the {@link getTags} method, tags are
     * added to a button's main container as CSS classes with the `toolwidget-tag--` prefix.
     *
     * For example, the `undo` tag would result in `toolwidget-tag--undo`
     * being added to the button's container's class list.
     *
     */
    setTags(tags: (string | ToolbarWidgetTag)[]): void;
    getTags(): string[];
    /**
     * Returns the ID of this widget in `container`. Adds a suffix to this' ID
     * if an item in `container` already has this' ID.
     *
     * For example, if `this` has ID `foo` and if
     * `container = { 'foo': somethingNotThis, 'foo-1': somethingElseNotThis }`, this method
     * returns `foo-2` because elements with IDs `foo` and `foo-1` are already present in
     * `container`.
     *
     * If `this` is already in `container`, returns the id given to `this` in the container.
     */
    getUniqueIdIn(container: Record<string, BaseWidget>): string;
    protected abstract getTitle(): string;
    protected abstract createIcon(): Element | null;
    protected fillDropdown(dropdown: HTMLElement, helpDisplay?: HelpDisplay): boolean;
    /**
     * Should return a 1-2 sentence description of the widget.
     *
     * At present, this is only used if this widget has an associated dropdown.
     */
    protected getHelpText(): undefined | string;
    /** @deprecated Renamed to `setUpButtonEventListeners`. */
    protected setupActionBtnClickListener(button: HTMLElement): void;
    protected setUpButtonEventListeners(button: HTMLElement): void;
    protected onKeyPress(_event: KeyPressEvent): boolean;
    protected abstract handleClick(): void;
    protected get hasDropdown(): boolean;
    protected addSubWidget(widget: BaseWidget): void;
    setLayoutManager(manager: WidgetContentLayoutManager): void;
    /**
     * Adds this to `parent`.
     * Returns the element that was just added to `parent`.
     * @internal
     */
    addTo(parent: HTMLElement): HTMLElement;
    /**
     * Remove this. This allows the widget to be added to a toolbar again
     * in the future using `addTo`.
     */
    remove(): void;
    focus(): void;
    /**
     * @internal
     */
    addCSSClassToContainer(className: string): void;
    removeCSSClassFromContainer(className: string): void;
    protected updateIcon(): void;
    setDisabled(disabled: boolean): void;
    setSelected(selected: boolean): void;
    protected setDropdownVisible(visible: boolean): void;
    /**
     * Only used by some layout managers.
     * In those layout managers, makes this dropdown visible.
     */
    protected activateDropdown(): void;
    /**
     * Returns `true` if this widget must always be in a toplevel menu and not
     * in a scrolling/overflow menu.
     *
     * This method can be overidden to override the default of `true`.
     */
    mustBeInToplevelMenu(): boolean;
    /**
     * Returns true iff this widget can be in a nontoplevel menu.
     *
     * @deprecated Use `!mustBeInToplevelMenu()` instead.
     */
    canBeInOverflowMenu(): boolean;
    getButtonWidth(): number;
    isHidden(): boolean;
    setHidden(hidden: boolean): void;
    /** Set whether the widget is contained within another. @internal */
    setIsToplevel(toplevel: boolean): void;
    /** Returns true if the menu for this widget is open. */
    protected isDropdownVisible(): boolean;
    protected isSelected(): boolean;
    private createDropdownIcon;
    /**
     * Serialize state associated with this widget.
     * Override this method to allow saving/restoring from state on application load.
     *
     * Overriders should call `super` and include the output of `super.serializeState` in
     * the output dictionary.
     *
     * Clients should not rely on the output from `saveState` being in any particular
     * format.
     */
    serializeState(): SavedToolbuttonState;
    /**
     * Restore widget state from serialized data. See also `saveState`.
     *
     * Overriders must call `super`.
     */
    deserializeFrom(state: SavedToolbuttonState): void;
}
