import { Color4 } from '@js-draw/math';
import TextRenderingStyle from '../rendering/TextRenderingStyle';
import { PenStyle } from '../tools/Pen';
import { EraserMode } from '../tools/Eraser';
import { SelectionMode } from '../tools/SelectionTool/types';
export type IconElemType = HTMLImageElement | SVGElement;
/**
 * Provides icons that can be used in the toolbar and other locations.
 *
 * To customize the icons used by the editor, extend this class and override methods.
 *
 * @example
 * ```ts,runnable
 * import * as jsdraw from 'js-draw';
 *
 * class CustomIconProvider extends jsdraw.IconProvider {
 *     // Use '☺' instead of the default dropdown symbol.
 *     public override makeDropdownIcon() {
 *         const icon = document.createElementNS('http://www.w3.org/2000/svg', 'svg');
 *         icon.innerHTML = `
 *             <text x='5' y='55' style='fill: var(--icon-color); font-size: 50pt;'>☺</text>
 *         `;
 *         icon.setAttribute('viewBox', '0 0 100 100');
 *         return icon;
 *     }
 * }
 *
 * const icons = new CustomIconProvider();
 * const editor = new jsdraw.Editor(document.body, {
 *     // The icon pack to use is specified through the editor's initial
 *     // configuration object:
 *     iconProvider: icons,
 * });
 *
 * // Add a toolbar that uses these icons
 * jsdraw.makeDropdownToolbar(editor).addDefaults();
 * ```
 */
export default class IconProvider {
    #private;
    makeUndoIcon(): IconElemType;
    makeRedoIcon(): IconElemType;
    makeDropdownIcon(): IconElemType;
    makeEraserIcon(eraserSize?: number, mode?: EraserMode): IconElemType;
    makeSelectionIcon(mode?: SelectionMode): IconElemType;
    makeRotateIcon(): IconElemType;
    makeHandToolIcon(): IconElemType;
    makeTouchPanningIcon(): IconElemType;
    /** Unused by js-draw. @deprecated */
    makeAllDevicePanningIcon(): IconElemType;
    makeZoomIcon(): IconElemType;
    makeRotationLockIcon(): IconElemType;
    makeInsertImageIcon(): IconElemType;
    makeUploadFileIcon(): IconElemType;
    makeTextIcon(textStyle: TextRenderingStyle): IconElemType;
    makePenIcon(penStyle: PenStyle): IconElemType;
    makeIconFromFactory(penStyle: PenStyle): IconElemType;
    makePipetteIcon(color?: Color4): IconElemType;
    makeShapeAutocorrectIcon(): IconElemType;
    makeStrokeSmoothingIcon(): IconElemType;
    makePressureSensitivityIcon(): IconElemType;
    /** Unused. @deprecated */
    makeFormatSelectionIcon(): IconElemType;
    makeResizeImageToSelectionIcon(): IconElemType;
    /** Renamed to {@link makeResizeImageToSelectionIcon} @deprecated */
    makeResizeViewportIcon(): IconElemType;
    makeDuplicateSelectionIcon(): IconElemType;
    makeCopyIcon(): IconElemType;
    makePasteIcon(): IconElemType;
    makeDeleteSelectionIcon(): IconElemType;
    makeCloseIcon(): IconElemType;
    makeSaveIcon(): IconElemType;
    makeConfigureDocumentIcon(): IconElemType;
    makeOverflowIcon(): IconElemType;
    makeHelpIcon(): IconElemType;
    /**
     * @param pathData - SVG path data (e.g. `m10,10l30,30z`)
     * @param fill - A valid CSS color (e.g. `var(--icon-color)` or `#f0f`). This can be `none`.
     */
    protected makeIconFromPath(pathData: string, fill?: string, strokeColor?: string, strokeWidth?: string): IconElemType;
    /**
     * @returns An object with both the definition of a checkerboard pattern and the syntax to
     * reference that pattern. The defs provided by this function should be wrapped within a
     * `<defs></defs>` element.
     *
     * **Note**: This function's return value includes both `patternDefElement` (which returns
     * an Element) and a (deprecated) `patternDef` string. Avoid using the `patternDef` result.
     */
    protected makeCheckerboardPattern(): {
        patternDefElement: SVGElement;
        readonly patternDef: string;
        patternRef: string;
    };
    /**
     * @returns true if the given `penStyle` is known to match a rounded tip type of pen.
     */
    protected isRoundedTipPen(penStyle: PenStyle): boolean;
    protected isPolylinePen(penStyle: PenStyle): boolean;
    /** Must be overridden by icon packs that need attribution. */
    licenseInfo(): string | null;
}
