import KeyBinding, { KeyCombination } from './KeyBinding';
type ShortcutDictionary = Record<string, KeyBinding[]>;
/**
 * Allows adding/changing keyboard shortcuts. This class provides static methods for registering
 * default shortcuts. An instance of this class must be used to access or change keyboard shortcuts.
 */
export default class KeyboardShortcutManager {
    private static shortcuts;
    private static shortcutDefaultDescriptions;
    private static shortcutLocalizedDescriptions;
    private shortcutOverrides;
    /**
     * Creates a new `ShortcutManager` with an initial set of shortcut overrides.
     *
     * @internal
     */
    constructor(initialOverrides: ShortcutDictionary);
    /**
     * Override an existing shortcut with a custom set of triggers.
     * @internal
     */
    overrideShortcut(shortcutId: string, overrideWith: KeyBinding[]): void;
    /** Returns true if `keyEvent` matches the shortcut with `shortcutId`. @internal */
    matchesShortcut(shortcutId: string, keyEvent: Partial<KeyCombination>): boolean;
    /**
     * Registers a default keyboard shortcut that can be overridden by individual instances
     * of `ShortcutManager`. Note that `id` should be a globally unique identifier.
     *
     * Only the first call to this method for a given `id` has an effect.
     *
     * @example
     * ```ts
     * const shortcutId = 'io.github.personalizedrefrigerator.js-draw.select-all';
     *
     * // Associate two shortcuts with the same ID
     * const shortcut1 = KeyBinding.fromString('ctrlOrMeta+a');
     * const shortcut2 = KeyBinding.fromString('ctrlOrMeta+shift+a');
     * KeyboardShortcutManager.registerDefaultKeyboardShortcut(
     * 	shortcutId,
     * 	[ shortcut1, shortcut2 ],
     * 	"Select All",
     * );
     *
     * // Provide a localized description
     * KeyboardShortcutManager.provideShortcutDescription(
     * 	shotcutId,
     * 	'es',
     * 	'Seleccionar todo',
     * );
     * ```
     *
     * @internal
     */
    static registerDefaultKeyboardShortcut(id: string, shortcuts: (KeyBinding | string)[], defaultDescription: string): boolean;
    /** Provides a localized description of a keyboard shortcut. @internal */
    static provideShortcutDescription(id: string, locale: string, description: string): void;
    /**
     * Gets all registered keyboard shortcut IDs.
     *
     * @see {@link getShortcutDescription}
     */
    static getAllShortcutIds(): string[];
    /**
     * Get the default keybindings associated with a keyboard shortcut.
     *
     * Any keybinding in the resultant list, by default, can trigger the function associated
     * with the shortcut.
     */
    static getShortcutDefaultKeybindings(shortcutId: string): KeyBinding[];
    /**
     * Get a description of a keyboard shortcut.
     *
     * `localeList`, if given, attempts to
     */
    static getShortcutDescription(id: string, localeList?: readonly string[]): string | null;
}
export {};
