import { Vec2 } from '@js-draw/math';
import  AbstractRenderer  from './AbstractRenderer.mjs';
// Renderer that outputs almost nothing. Useful for automated tests.
export default class DummyRenderer extends AbstractRenderer {
    constructor(viewport) {
        super(viewport);
        // Variables that track the state of what's been rendered
        this.clearedCount = 0;
        this.renderedPathCount = 0;
        this.lastFillStyle = null;
        this.lastPoint = null;
        this.objectNestingLevel = 0;
        this.lastText = null;
        this.lastImage = null;
        // List of points drawn since the last clear.
        this.pointBuffer = [];
    }
    displaySize() {
        // Do we have a stored viewport size?
        const viewportSize = this.getViewport().getScreenRectSize();
        // Don't use a 0x0 viewport — DummyRenderer is often used
        // for tests that run without a display, so pretend we have a
        // reasonable-sized display.
        if (viewportSize.x === 0 || viewportSize.y === 0) {
            return Vec2.of(640, 480);
        }
        return viewportSize;
    }
    clear() {
        this.clearedCount++;
        this.renderedPathCount = 0;
        this.pointBuffer = [];
        this.lastText = null;
        this.lastImage = null;
        // Ensure all objects finished rendering
        if (this.objectNestingLevel > 0) {
            throw new Error(`Within an object while clearing! Nesting level: ${this.objectNestingLevel}`);
        }
    }
    beginPath(startPoint) {
        this.lastPoint = startPoint;
        this.pointBuffer.push(startPoint);
    }
    endPath(style) {
        this.renderedPathCount++;
        this.lastFillStyle = style;
    }
    lineTo(point) {
        point = this.canvasToScreen(point);
        this.lastPoint = point;
        this.pointBuffer.push(point);
    }
    moveTo(point) {
        point = this.canvasToScreen(point);
        this.lastPoint = point;
        this.pointBuffer.push(point);
    }
    traceCubicBezierCurve(p1, p2, p3) {
        p1 = this.canvasToScreen(p1);
        p2 = this.canvasToScreen(p2);
        p3 = this.canvasToScreen(p3);
        this.lastPoint = p3;
        this.pointBuffer.push(p1, p2, p3);
    }
    traceQuadraticBezierCurve(controlPoint, endPoint) {
        controlPoint = this.canvasToScreen(controlPoint);
        endPoint = this.canvasToScreen(endPoint);
        this.lastPoint = endPoint;
        this.pointBuffer.push(controlPoint, endPoint);
    }
    drawPoints(..._points) {
        // drawPoints is intended for debugging.
        // As such, it is unlikely to be the target of automated tests.
    }
    drawText(text, _transform, _style) {
        this.lastText = text;
    }
    drawImage(image) {
        this.lastImage = image;
    }
    startObject(boundingBox, _clip) {
        super.startObject(boundingBox);
        this.objectNestingLevel += 1;
    }
    endObject() {
        super.endObject();
        this.objectNestingLevel -= 1;
    }
    isTooSmallToRender(_rect) {
        return false;
    }
    canRenderFromWithoutDataLoss(other) {
        return other instanceof DummyRenderer;
    }
    renderFromOtherOfSameType(transform, other) {
        if (!(other instanceof DummyRenderer)) {
            throw new Error(`${other} cannot be rendered onto ${this}`);
        }
        this.renderedPathCount += other.renderedPathCount;
        this.lastFillStyle = other.lastFillStyle;
        this.lastPoint = other.lastPoint;
        this.pointBuffer.push(...other.pointBuffer.map((point) => {
            return transform.transformVec2(point);
        }));
    }
    toString() {
        return '[DummyRenderer]';
    }
}
