//
// Used by `SVGLoader`s to store unrecognised global attributes
// (e.g. unrecognised XML namespace declarations).
// @internal
// @packageDocumentation
//
import { Rect2 } from '@js-draw/math';
import  SVGRenderer  from '../rendering/renderers/SVGRenderer.mjs';
import  AbstractComponent, { ComponentSizingMode }  from './AbstractComponent.mjs';
const componentKind = 'svg-global-attributes';
// Stores global SVG attributes (e.g. namespace identifiers.)
export default class SVGGlobalAttributesObject extends AbstractComponent {
    // Does not modify `attrs`
    constructor(attrs) {
        super(componentKind);
        this.contentBBox = Rect2.empty;
        // Already stored/managed in `editor.image`.
        const attrsManagedByRenderer = ['viewBox', 'width', 'height'];
        // Only store attributes that aren't managed by other parts of the app.
        this.attrs = attrs.filter(([attr, _value]) => {
            return !attrsManagedByRenderer.includes(attr);
        });
    }
    render(canvas, _visibleRect) {
        if (!(canvas instanceof SVGRenderer)) {
            // Don't draw unrenderable objects if we can't
            return;
        }
        for (const [attr, value] of this.attrs) {
            canvas.setRootSVGAttribute(attr, value);
        }
    }
    intersects(_lineSegment) {
        return false;
    }
    applyTransformation(_affineTransfm) { }
    isSelectable() {
        return false;
    }
    getSizingMode() {
        // This component can be shown anywhere (it won't be
        // visible to the user, it just needs to be saved with
        // the image).
        return ComponentSizingMode.Anywhere;
    }
    createClone() {
        return new SVGGlobalAttributesObject(this.attrs);
    }
    description(localization) {
        return localization.svgObject;
    }
    serializeToJSON() {
        return JSON.stringify(this.attrs);
    }
    static deserializeFromString(_data) {
        // To be safe, don't deserialize any attributes
        return new SVGGlobalAttributesObject([]);
    }
}
AbstractComponent.registerComponent(componentKind, SVGGlobalAttributesObject.deserializeFromString);
