import { Color4 } from '@js-draw/math';
import  SerializableCommand  from '../commands/SerializableCommand.mjs';
import  UnresolvedSerializableCommand  from '../commands/UnresolvedCommand.mjs';
import  { textStyleFromJSON, textStyleToJSON, }  from '../rendering/TextRenderingStyle.mjs';
const serializeComponentStyle = (style) => {
    const result = {};
    if (style.color) {
        result.color = style.color.toHexString();
    }
    if (style.textStyle) {
        result.textStyle = textStyleToJSON(style.textStyle);
    }
    return result;
};
const deserializeComponentStyle = (json) => {
    const color = json.color ? Color4.fromHex(json.color) : undefined;
    const textStyle = json.textStyle ? textStyleFromJSON(json.textStyle) : undefined;
    return {
        color,
        textStyle,
    };
};
// For internal use by Components implementing `updateStyle`:
export const createRestyleComponentCommand = (initialStyle, newStyle, component) => {
    return new DefaultRestyleComponentCommand(initialStyle, newStyle, component.getId(), component);
};
// Returns true if `component` is a `RestylableComponent`.
export const isRestylableComponent = (component) => {
    const hasMethods = 'getStyle' in component && 'updateStyle' in component && 'forceStyle' in component;
    if (!hasMethods) {
        return false;
    }
    if (!('isRestylableComponent' in component) || !component['isRestylableComponent']) {
        return false;
    }
    return true;
};
const defaultRestyleComponentCommandId = 'default-restyle-element';
class DefaultRestyleComponentCommand extends UnresolvedSerializableCommand {
    constructor(originalStyle, newStyle, componentID, component) {
        super(defaultRestyleComponentCommandId, componentID, component);
        this.originalStyle = originalStyle;
        this.newStyle = newStyle;
    }
    getComponent(editor) {
        this.resolveComponent(editor.image);
        const component = this.component;
        if (!component || !component['forceStyle'] || !component['updateStyle']) {
            throw new Error('this.component is missing forceStyle and/or updateStyle methods!');
        }
        return component;
    }
    apply(editor) {
        this.getComponent(editor).forceStyle(this.newStyle, editor);
    }
    unapply(editor) {
        this.getComponent(editor).forceStyle(this.originalStyle, editor);
    }
    description(editor, localizationTable) {
        return localizationTable.restyledElement(this.getComponent(editor).description(localizationTable));
    }
    serializeToJSON() {
        return {
            id: this.componentID,
            originalStyle: serializeComponentStyle(this.originalStyle),
            newStyle: serializeComponentStyle(this.newStyle),
        };
    }
}
(() => {
    SerializableCommand.register(defaultRestyleComponentCommandId, (json, _editor) => {
        const origStyle = deserializeComponentStyle(json.originalStyle);
        const newStyle = deserializeComponentStyle(json.newStyle);
        const id = json.id;
        if (typeof json.id !== 'string') {
            throw new Error(`json.id is of type ${typeof json.id}, not string.`);
        }
        return new DefaultRestyleComponentCommand(origStyle, newStyle, id);
    });
})();
