import  Command  from './Command.mjs';
import  SerializableCommand  from './SerializableCommand.mjs';
// Returns a command that does the opposite of the given command --- `result.apply()` calls
// `command.unapply()` and `result.unapply()` calls `command.apply()`.
const invertCommand = (command) => {
    if (command instanceof SerializableCommand) {
        // SerializableCommand that does the inverse of [command]
        return new (class extends SerializableCommand {
            constructor() {
                super(...arguments);
                // For debugging
                this._command = command;
            }
            serializeToJSON() {
                return command.serialize();
            }
            apply(editor) {
                command.unapply(editor);
            }
            unapply(editor) {
                command.apply(editor);
            }
            onDrop(editor) {
                command.onDrop(editor);
            }
            description(editor, localizationTable) {
                return localizationTable.inverseOf(command.description(editor, localizationTable));
            }
        })('inverse');
    }
    else {
        // Command that does the inverse of [command].
        const result = new (class extends Command {
            apply(editor) {
                command.unapply(editor);
            }
            unapply(editor) {
                command.apply(editor);
            }
            onDrop(editor) {
                command.onDrop(editor);
            }
            description(editor, localizationTable) {
                return localizationTable.inverseOf(command.description(editor, localizationTable));
            }
        })();
        // We know that T does not extend SerializableCommand, and thus returning a Command
        // is appropriate.
        return result;
    }
};
SerializableCommand.register('inverse', (data, editor) => {
    return invertCommand(SerializableCommand.deserialize(data, editor));
});
export default invertCommand;
