import AbstractComponent from '../components/AbstractComponent';
import Editor from '../Editor';
import { EditorLocalization } from '../localization';
import SerializableCommand from './SerializableCommand';
/**
 * Removes the given {@link AbstractComponent}s from the image.
 *
 * **Example**:
 * ```ts,runnable
 * import { Editor, Erase, uniteCommands, Color4, Path, Stroke, Rect2, pathToRenderable } from 'js-draw';
 *
 * const editor = new Editor(document.body);
 * editor.addToolbar();
 *
 * // Add a large number of strokes
 * const commands = [];
 * for (let x = -20; x < 20; x++) {
 *   for (let y = 0; y < 60; y++) {
 *     const stroke = new Stroke([
 *       pathToRenderable(
 *         Path.fromString(`m${x * 5},${y * 5}l1,1`),
 *         { fill: Color4.transparent, stroke: {width: 2, color: Color4.ofRGB(x / 10, y / 10, 0.5)}} )
 *       ]);
 *     commands.push(editor.image.addElement(stroke));
 *   }
 * }
 * await editor.dispatch(uniteCommands(commands, 100));
 *
 * ---visible---
 * // Given some editor...
 *
 * // Find all elements intersecting the rectangle with top left (-10,-30) and
 * // (width,height)=(50,100).
 * const elems = editor.image.getComponentsIntersecting(
 * 	new Rect2(-10, -30, 50, 100)
 * );
 *
 * // Create a command that erases [elems] when applied
 * const eraseElemsCmd = new Erase(elems);
 *
 * // Apply the command (and make it undoable)
 * editor.dispatch(eraseElemsCmd);
 * ```
 */
export default class Erase extends SerializableCommand {
    private toRemove;
    private applied;
    constructor(toRemove: AbstractComponent[]);
    apply(editor: Editor): void;
    unapply(editor: Editor): void;
    onDrop(editor: Editor): void;
    description(_editor: Editor, localizationTable: EditorLocalization): string;
    protected serializeToJSON(): {
        name: string;
        zIndex: number;
        id: string;
        loadSaveData: import("../components/AbstractComponent").LoadSaveDataTable;
        data: string | number | any[] | Record<string, any>;
    }[];
}
