type ListenerResult = {
    remove(): void;
};
type UpdateCallback<T> = (value: T) => void;
type ReactiveValuesOf<T extends unknown[]> = {
    [key in keyof T]: ReactiveValue<T[key]>;
};
/**
 * A `ReactiveValue` is a value that
 * - updates periodically,
 * - can fire listeners when it updates,
 * - and can be chanined together with other `ReactiveValue`s.
 *
 * A `ReactiveValue` is a read-only view. See {@link MutableReactiveValue} for a
 * read-write view.
 *
 * Static methods in the `ReactiveValue` and `MutableReactiveValue` classes are
 * constructors (e.g. `fromImmutable`).
 *
 * Avoid extending this class from an external library, as that may not be stable.
 */
export declare abstract class ReactiveValue<T> {
    /**
     * Returns a reference to the current value of this `ReactiveValue`.
     *
     * The result of this **should not be modified** (use `setValue` instead).
     */
    abstract get(): T;
    /**
     * Registers a listener that is notified when the value of this changes.
     */
    abstract onUpdate(listener: UpdateCallback<T>): ListenerResult;
    /**
     * Calls `callback` immediately, then registers `callback` as an onUpdateListener.
     *
     * @see {@link onUpdate}.
     */
    abstract onUpdateAndNow(callback: UpdateCallback<T>): ListenerResult;
    /** Returns a promise that resolves when this value is next changed. */
    waitForNextUpdate(): Promise<T>;
    /** Creates a `ReactiveValue` with an initial value, `initialValue`. */
    static fromInitialValue<T>(initialValue: T): MutableReactiveValue<T>;
    /** Returns a `ReactiveValue` that is **known** will never change. */
    static fromImmutable<T>(value: T): ReactiveValue<T>;
    /**
     * Creates a `ReactiveValue` whose values come from `callback`.
     *
     * `callback` is called whenever any of `sourceValues` are updated and initially to
     * set the initial value of the result.
     */
    static fromCallback<T>(callback: () => T, sourceValues: ReactiveValue<any>[]): ReactiveValue<T>;
    /**
     * Returns a reactive value derived from a single `source`.
     *
     * If `inverseMap` is `undefined`, the result is a read-only view.
     */
    static map<A, B>(source: ReactiveValue<A>, map: (a: A) => B, inverseMap?: undefined): ReactiveValue<B>;
    /**
     * Returns a reactive value derived from a single `source`.
     */
    static map<A, B>(source: ReactiveValue<A>, map: (a: A) => B, inverseMap: (b: B) => A): MutableReactiveValue<B>;
    static union<Values extends [...unknown[]]>(values: ReactiveValuesOf<Values>): ReactiveValue<Values>;
}
export declare abstract class MutableReactiveValue<T> extends ReactiveValue<T> {
    /**
     * Changes the value of this and, if different, fires all update listeners.
     *
     * @see {@link onUpdate}
     */
    abstract set(newValue: T): void;
    static fromProperty<SourceType extends object, Name extends keyof SourceType>(sourceValue: MutableReactiveValue<SourceType>, propertyName: Name): MutableReactiveValue<SourceType[Name]>;
}
export default ReactiveValue;
