"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const TextComponent_1 = __importDefault(require("../components/TextComponent"));
const EditorImage_1 = __importDefault(require("../image/EditorImage"));
const math_1 = require("@js-draw/math");
const Pointer_1 = require("../Pointer");
const types_1 = require("../types");
const BaseTool_1 = __importDefault(require("./BaseTool"));
const Erase_1 = __importDefault(require("../commands/Erase"));
const uniteCommands_1 = __importDefault(require("../commands/uniteCommands"));
const ReactiveValue_1 = require("../util/ReactiveValue");
const overlayCSSClass = 'textEditorOverlay';
/** A tool that allows users to enter and edit text. */
class TextTool extends BaseTool_1.default {
    constructor(editor, description, localizationTable) {
        super(editor.notifier, description);
        this.editor = editor;
        this.localizationTable = localizationTable;
        this.textInputElem = null;
        this.textMeasuringCtx = null;
        this.removeExistingCommand = null;
        const editorFonts = editor.getCurrentSettings().text?.fonts ?? [];
        this.textStyleValue = ReactiveValue_1.ReactiveValue.fromInitialValue({
            size: 32,
            fontFamily: editorFonts.length > 0 ? editorFonts[0] : 'sans-serif',
            renderingStyle: {
                fill: math_1.Color4.purple,
            },
        });
        this.textStyleValue.onUpdateAndNow(() => {
            this.textStyle = this.textStyleValue.get();
            this.updateTextInput();
            this.editor.notifier.dispatch(types_1.EditorEventType.ToolUpdated, {
                kind: types_1.EditorEventType.ToolUpdated,
                tool: this,
            });
        });
        this.contentTransform = ReactiveValue_1.ReactiveValue.fromInitialValue(math_1.Mat33.identity);
        this.textEditOverlay = document.createElement('div');
        this.textEditOverlay.classList.add(overlayCSSClass);
        this.editor.addStyleSheet(`
			.${overlayCSSClass} textarea {
				background-color: rgba(0, 0, 0, 0);

				white-space: pre;
				overflow: hidden;

				padding: 0;
				margin: 0;
				border: none;
				padding: 0;

				min-width: 100px;
				min-height: 1.1em;
			}
		`);
        this.anchorControl = this.editor.anchorElementToCanvas(this.textEditOverlay, this.contentTransform);
    }
    initTextMeasuringCanvas() {
        this.textMeasuringCtx ??= document.createElement('canvas').getContext('2d');
    }
    getTextAscent(text, style) {
        this.initTextMeasuringCanvas();
        if (this.textMeasuringCtx) {
            this.textMeasuringCtx.textBaseline = 'alphabetic';
            TextComponent_1.default.applyTextStyles(this.textMeasuringCtx, style);
            const measurement = this.textMeasuringCtx.measureText(text);
            return measurement.fontBoundingBoxAscent ?? measurement.actualBoundingBoxAscent;
        }
        // Estimate
        return (style.size * 2) / 3;
    }
    // Take input from this' textInputElem and add it to the EditorImage.
    // If [removeInput], the HTML input element is removed. Otherwise, its value
    // is cleared.
    flushInput(removeInput = true) {
        if (!this.textInputElem)
            return;
        // Determine the scroll first -- removing the input (and other DOM changes)
        // also change the scroll.
        const scrollingRegion = this.textEditOverlay.parentElement;
        const containerScroll = math_1.Vec2.of(scrollingRegion?.scrollLeft ?? 0, scrollingRegion?.scrollTop ?? 0);
        const content = this.textInputElem.value.trimEnd();
        this.textInputElem.value = '';
        if (removeInput) {
            // In some browsers, .remove() triggers a .blur event (synchronously).
            // Clear this.textInputElem before removal
            const input = this.textInputElem;
            this.textInputElem = null;
            input.remove();
        }
        if (content !== '') {
            // When the text is long, it can cause its container to scroll so that the
            // editing caret is in view.
            // So that the text added to the document is in the same position as the text
            // shown in the editor, account for this scroll when computing the transform:
            const scrollCorrectionScreen = containerScroll.times(-1);
            // Uses .transformVec3 to avoid also translating the scroll correction (treating
            // it as a point):
            const scrollCorrectionCanvas = this.editor.viewport.screenToCanvasTransform.transformVec3(scrollCorrectionScreen);
            const scrollTransform = math_1.Mat33.translation(scrollCorrectionCanvas);
            const textComponent = TextComponent_1.default.fromLines(content.split('\n'), scrollTransform.rightMul(this.contentTransform.get()), this.textStyle);
            const action = EditorImage_1.default.addComponent(textComponent);
            if (this.removeExistingCommand) {
                // Unapply so that `removeExistingCommand` can be added to the undo stack.
                this.removeExistingCommand.unapply(this.editor);
                this.editor.dispatch((0, uniteCommands_1.default)([this.removeExistingCommand, action]));
                this.removeExistingCommand = null;
            }
            else {
                this.editor.dispatch(action);
            }
        }
    }
    updateTextInput() {
        if (!this.textInputElem) {
            return;
        }
        this.textInputElem.placeholder = this.localizationTable.enterTextToInsert;
        this.textInputElem.style.fontFamily = this.textStyle.fontFamily;
        this.textInputElem.style.fontStyle = this.textStyle.fontStyle ?? '';
        this.textInputElem.style.fontVariant = this.textStyle.fontVariant ?? '';
        this.textInputElem.style.fontWeight = this.textStyle.fontWeight ?? '';
        this.textInputElem.style.fontSize = `${this.textStyle.size}px`;
        this.textInputElem.style.color = this.textStyle.renderingStyle.fill.toHexString();
        this.textInputElem.style.margin = '0';
        this.textInputElem.style.width = `${this.textInputElem.scrollWidth}px`;
        this.textInputElem.style.height = `${this.textInputElem.scrollHeight}px`;
        // Get the ascent based on the font, using a string of characters
        // that is tall in most fonts.
        const tallText = 'Testing!';
        const ascent = this.getTextAscent(tallText, this.textStyle);
        const vertAdjust = ascent;
        this.textInputElem.style.transform = `translate(0, ${-vertAdjust}px)`;
        this.textInputElem.style.transformOrigin = 'top left';
        // Match the line height of default rendered text.
        const lineHeight = Math.floor(this.textStyle.size);
        this.textInputElem.style.lineHeight = `${lineHeight}px`;
    }
    startTextInput(textCanvasPos, initialText) {
        this.flushInput();
        this.textInputElem = document.createElement('textarea');
        this.textInputElem.value = initialText;
        this.textInputElem.style.display = 'inline-block';
        const textTargetPosition = this.editor.viewport.roundPoint(textCanvasPos);
        const textRotation = -this.editor.viewport.getRotationAngle();
        const textScale = math_1.Vec2.of(1, 1).times(this.editor.viewport.getSizeOfPixelOnCanvas());
        this.contentTransform.set(
        // Scale, then rotate, then translate:
        math_1.Mat33.translation(textTargetPosition)
            .rightMul(math_1.Mat33.zRotation(textRotation))
            .rightMul(math_1.Mat33.scaling2D(textScale)));
        this.updateTextInput();
        // Update the input size/position/etc. after the placeHolder has had time to appear.
        setTimeout(() => this.updateTextInput(), 0);
        this.textInputElem.oninput = () => {
            if (this.textInputElem) {
                this.textInputElem.style.width = `${this.textInputElem.scrollWidth}px`;
                this.textInputElem.style.height = `${this.textInputElem.scrollHeight}px`;
            }
        };
        this.textInputElem.onblur = () => {
            const input = this.textInputElem;
            // Delay removing the input -- flushInput may be called within a blur()
            // event handler
            const removeInput = false;
            this.flushInput(removeInput);
            this.textInputElem = null;
            if (input) {
                input.classList.add('-hiding');
            }
            setTimeout(() => {
                input?.remove();
            }, 0);
        };
        this.textInputElem.onkeyup = (evt) => {
            // In certain input modes, the <enter> key is used to select characters.
            // When in this mode, prevent <enter> from submitting:
            if (evt.isComposing)
                return;
            if (evt.key === 'Enter' && !evt.shiftKey) {
                this.flushInput();
                this.editor.focus();
            }
            else if (evt.key === 'Escape') {
                // Cancel input.
                this.textInputElem?.remove();
                this.textInputElem = null;
                this.editor.focus();
                this.removeExistingCommand?.unapply(this.editor);
                this.removeExistingCommand = null;
            }
        };
        this.textEditOverlay.replaceChildren(this.textInputElem);
        setTimeout(() => this.textInputElem?.focus(), 0);
    }
    setEnabled(enabled) {
        super.setEnabled(enabled);
        if (!this.isEnabled()) {
            this.flushInput();
        }
        this.textEditOverlay.style.display = enabled ? 'block' : 'none';
    }
    onPointerDown({ current, allPointers }) {
        if (current.device === Pointer_1.PointerDevice.Eraser) {
            return false;
        }
        if (allPointers.length === 1) {
            // Are we clicking on a text node?
            const canvasPos = current.canvasPos;
            const halfTestRegionSize = math_1.Vec2.of(4, 4).times(this.editor.viewport.getSizeOfPixelOnCanvas());
            const testRegion = math_1.Rect2.fromCorners(canvasPos.minus(halfTestRegionSize), canvasPos.plus(halfTestRegionSize));
            const targetNodes = this.editor.image.getComponentsIntersecting(testRegion);
            let targetTextNodes = targetNodes.filter((node) => node instanceof TextComponent_1.default);
            // Don't try to edit text nodes that contain the viewport (this allows us
            // to zoom in on text nodes and add text on top of them.)
            const visibleRect = this.editor.viewport.visibleRect;
            targetTextNodes = targetTextNodes.filter((node) => !node.getBBox().containsRect(visibleRect));
            // End any TextNodes we're currently editing.
            this.flushInput();
            if (targetTextNodes.length > 0) {
                const targetNode = targetTextNodes[targetTextNodes.length - 1];
                this.setTextStyle(targetNode.getTextStyle());
                // Create and temporarily apply removeExistingCommand.
                this.removeExistingCommand = new Erase_1.default([targetNode]);
                this.removeExistingCommand.apply(this.editor);
                this.startTextInput(targetNode.getBaselinePos(), targetNode.getText());
                this.contentTransform.set(targetNode.getTransform());
                this.updateTextInput();
            }
            else {
                this.removeExistingCommand = null;
                this.startTextInput(current.canvasPos, '');
            }
            return true;
        }
        return false;
    }
    onGestureCancel() {
        this.flushInput();
        this.editor.focus();
    }
    setFontFamily(fontFamily) {
        if (fontFamily !== this.textStyle.fontFamily) {
            this.textStyleValue.set({
                ...this.textStyle,
                fontFamily: fontFamily,
            });
        }
    }
    setColor(color) {
        if (!color.eq(this.textStyle.renderingStyle.fill)) {
            this.textStyleValue.set({
                ...this.textStyle,
                renderingStyle: {
                    ...this.textStyle.renderingStyle,
                    fill: color,
                },
            });
        }
    }
    setFontSize(size) {
        if (size !== this.textStyle.size) {
            this.textStyleValue.set({
                ...this.textStyle,
                size,
            });
        }
    }
    getTextStyle() {
        return this.textStyle;
    }
    getStyleValue() {
        return this.textStyleValue;
    }
    setTextStyle(style) {
        this.textStyleValue.set(style);
    }
    // @internal
    onDestroy() {
        super.onDestroy();
        this.anchorControl.remove();
    }
}
exports.default = TextTool;
