"use strict";
// @internal @packageDocumentation
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const BaseTool_1 = __importDefault(require("./BaseTool"));
/**
 * A tool used internally to pick colors from the canvas.
 *
 * When color selection is in progress, the `pipette--color-selection-in-progress` class
 * is added to the root element. This can be used by themes.
 *
 * @internal
 */
class PipetteTool extends BaseTool_1.default {
    constructor(editor, description) {
        super(editor.notifier, description);
        this.editor = editor;
        this.colorPreviewListener = null;
        this.colorSelectListener = null;
        this.enabledValue().onUpdateAndNow(() => {
            this.updateSelectingStatus();
        });
    }
    canReceiveInputInReadOnlyEditor() {
        return true;
    }
    // Ensures that the root editor element correctly reflects whether color selection
    // is in progress.
    updateSelectingStatus() {
        const className = 'pipette--color-selection-in-progress';
        if (this.isEnabled() && this.colorSelectListener && this.colorPreviewListener) {
            this.editor.getRootElement().classList.add(className);
        }
        else {
            this.editor.getRootElement().classList.remove(className);
        }
    }
    setColorListener(colorPreviewListener, 
    // Called when the gesture ends -- when the user has selected a color.
    colorSelectListener) {
        this.colorPreviewListener = colorPreviewListener;
        this.colorSelectListener = colorSelectListener;
        this.updateSelectingStatus();
    }
    clearColorListener() {
        this.colorPreviewListener = null;
        this.colorSelectListener = null;
        this.updateSelectingStatus();
    }
    onPointerDown({ current, allPointers }) {
        if (this.colorPreviewListener && allPointers.length === 1) {
            this.colorPreviewListener(this.editor.display.getColorAt(current.screenPos));
            return true;
        }
        return false;
    }
    onPointerMove({ current }) {
        this.colorPreviewListener?.(this.editor.display.getColorAt(current.screenPos));
    }
    onPointerUp({ current }) {
        this.colorSelectListener?.(this.editor.display.getColorAt(current.screenPos));
    }
    onGestureCancel() {
        this.colorSelectListener?.(null);
    }
}
exports.default = PipetteTool;
