"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const math_1 = require("@js-draw/math");
const types_1 = require("../../types");
const constants_1 = require("../constants");
const makeColorInput_1 = __importDefault(require("./components/makeColorInput"));
const BaseToolWidget_1 = __importDefault(require("./BaseToolWidget"));
class TextToolWidget extends BaseToolWidget_1.default {
    constructor(editor, tool, localization) {
        super(editor, tool, 'text-tool-widget', localization);
        this.tool = tool;
        this.updateDropdownInputs = null;
        editor.notifier.on(types_1.EditorEventType.ToolUpdated, (evt) => {
            if (evt.kind === types_1.EditorEventType.ToolUpdated && evt.tool === tool) {
                this.updateIcon();
                this.updateDropdownInputs?.();
            }
        });
    }
    getTitle() {
        return this.targetTool.description;
    }
    createIcon() {
        const textStyle = this.tool.getTextStyle();
        return this.editor.icons.makeTextIcon(textStyle);
    }
    fillDropdown(dropdown) {
        const container = document.createElement('div');
        container.classList.add(`${constants_1.toolbarCSSPrefix}spacedList`, `${constants_1.toolbarCSSPrefix}nonbutton-controls-main-list`);
        const fontRow = document.createElement('div');
        const colorRow = document.createElement('div');
        const sizeRow = document.createElement('div');
        const fontInput = document.createElement('select');
        const fontLabel = document.createElement('label');
        const sizeInput = document.createElement('input');
        const sizeLabel = document.createElement('label');
        const { input: colorInput, container: colorInputContainer, setValue: setColorInputValue, } = (0, makeColorInput_1.default)(this.editor, (color) => {
            this.tool.setColor(color);
        });
        const colorLabel = document.createElement('label');
        const fontsInInput = new Set();
        const addFontToInput = (fontName) => {
            const option = document.createElement('option');
            option.value = fontName;
            option.textContent = fontName;
            fontInput.appendChild(option);
            fontsInInput.add(fontName);
        };
        sizeInput.setAttribute('type', 'number');
        sizeInput.min = '1';
        sizeInput.max = '128';
        fontLabel.innerText = this.localizationTable.fontLabel;
        colorLabel.innerText = this.localizationTable.colorLabel;
        sizeLabel.innerText = this.localizationTable.textSize;
        colorInput.id = `${constants_1.toolbarCSSPrefix}-text-color-input-${TextToolWidget.idCounter++}`;
        colorLabel.setAttribute('for', colorInput.id);
        sizeInput.id = `${constants_1.toolbarCSSPrefix}-text-size-input-${TextToolWidget.idCounter++}`;
        sizeLabel.setAttribute('for', sizeInput.id);
        const defaultFonts = this.editor.getCurrentSettings().text?.fonts ?? [];
        for (const font of defaultFonts) {
            addFontToInput(font);
        }
        fontInput.classList.add('font-selector');
        fontInput.id = `${constants_1.toolbarCSSPrefix}-text-font-input-${TextToolWidget.idCounter++}`;
        fontLabel.setAttribute('for', fontInput.id);
        fontInput.onchange = () => {
            this.tool.setFontFamily(fontInput.value);
        };
        sizeInput.onchange = () => {
            const size = parseInt(sizeInput.value);
            if (!isNaN(size) && size > 0) {
                this.tool.setFontSize(size);
            }
        };
        colorRow.appendChild(colorLabel);
        colorRow.appendChild(colorInputContainer);
        fontRow.appendChild(fontLabel);
        fontRow.appendChild(fontInput);
        sizeRow.appendChild(sizeLabel);
        sizeRow.appendChild(sizeInput);
        this.updateDropdownInputs = () => {
            const style = this.tool.getTextStyle();
            setColorInputValue(style.renderingStyle.fill);
            if (!fontsInInput.has(style.fontFamily)) {
                addFontToInput(style.fontFamily);
            }
            fontInput.value = style.fontFamily;
            sizeInput.value = `${style.size}`;
        };
        this.updateDropdownInputs();
        container.replaceChildren(colorRow, sizeRow, fontRow);
        dropdown.appendChild(container);
        return true;
    }
    serializeState() {
        const textStyle = this.tool.getTextStyle();
        return {
            ...super.serializeState(),
            fontFamily: textStyle.fontFamily,
            textSize: textStyle.size,
            color: textStyle.renderingStyle.fill.toHexString(),
        };
    }
    deserializeFrom(state) {
        if (state.fontFamily && typeof state.fontFamily === 'string') {
            this.tool.setFontFamily(state.fontFamily);
        }
        if (state.color && typeof state.color === 'string') {
            this.tool.setColor(math_1.Color4.fromHex(state.color));
        }
        if (state.textSize && typeof state.textSize === 'number') {
            this.tool.setFontSize(state.textSize);
        }
        super.deserializeFrom(state);
    }
}
TextToolWidget.idCounter = 0;
exports.default = TextToolWidget;
