"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const ArrowBuilder_1 = require("../../components/builders/ArrowBuilder");
const FreehandLineBuilder_1 = require("../../components/builders/FreehandLineBuilder");
const PressureSensitiveFreehandLineBuilder_1 = require("../../components/builders/PressureSensitiveFreehandLineBuilder");
const LineBuilder_1 = require("../../components/builders/LineBuilder");
const RectangleBuilder_1 = require("../../components/builders/RectangleBuilder");
const CircleBuilder_1 = require("../../components/builders/CircleBuilder");
const types_1 = require("../../types");
const makeColorInput_1 = __importDefault(require("./components/makeColorInput"));
const BaseToolWidget_1 = __importDefault(require("./BaseToolWidget"));
const math_1 = require("@js-draw/math");
const keybindings_1 = require("./keybindings");
const constants_1 = require("../constants");
const makeThicknessSlider_1 = __importDefault(require("./components/makeThicknessSlider"));
const makeGridSelector_1 = __importDefault(require("./components/makeGridSelector"));
const PolylineBuilder_1 = require("../../components/builders/PolylineBuilder");
const createElement_1 = __importDefault(require("../../util/dom/createElement"));
/**
 * This toolbar widget allows a user to control a single {@link Pen} tool.
 *
 * See also {@link AbstractToolbar.addDefaultToolWidgets}.
 */
class PenToolWidget extends BaseToolWidget_1.default {
    constructor(editor, tool, localization) {
        super(editor, tool, 'pen', localization);
        this.tool = tool;
        this.updateInputs = () => { };
        // Pen types that correspond to
        this.shapelikeIDs = ['pressure-sensitive-pen', 'freehand-pen'];
        // Additional client-specified pens.
        const additionalPens = editor.getCurrentSettings().pens?.additionalPenTypes ?? [];
        const filterPens = editor.getCurrentSettings().pens?.filterPenTypes ?? (() => true);
        // Default pen types
        this.penTypes = [
            // Non-shape pens
            {
                name: this.localizationTable.flatTipPen,
                id: 'pressure-sensitive-pen',
                factory: PressureSensitiveFreehandLineBuilder_1.makePressureSensitiveFreehandLineBuilder,
            },
            {
                name: this.localizationTable.roundedTipPen,
                id: 'freehand-pen',
                factory: FreehandLineBuilder_1.makeFreehandLineBuilder,
            },
            {
                name: this.localizationTable.roundedTipPen2,
                id: 'polyline-pen',
                factory: PolylineBuilder_1.makePolylineBuilder,
            },
            ...additionalPens.filter((pen) => !pen.isShapeBuilder),
            // Shape pens
            {
                name: this.localizationTable.arrowPen,
                id: 'arrow',
                isShapeBuilder: true,
                factory: ArrowBuilder_1.makeArrowBuilder,
            },
            {
                name: this.localizationTable.linePen,
                id: 'line',
                isShapeBuilder: true,
                factory: LineBuilder_1.makeLineBuilder,
            },
            {
                name: this.localizationTable.filledRectanglePen,
                id: 'filled-rectangle',
                isShapeBuilder: true,
                factory: RectangleBuilder_1.makeFilledRectangleBuilder,
            },
            {
                name: this.localizationTable.outlinedRectanglePen,
                id: 'outlined-rectangle',
                isShapeBuilder: true,
                factory: RectangleBuilder_1.makeOutlinedRectangleBuilder,
            },
            {
                name: this.localizationTable.outlinedCirclePen,
                id: 'outlined-circle',
                isShapeBuilder: true,
                factory: CircleBuilder_1.makeOutlinedCircleBuilder,
            },
            ...additionalPens.filter((pen) => pen.isShapeBuilder),
        ].filter(filterPens);
        this.editor.notifier.on(types_1.EditorEventType.ToolUpdated, (toolEvt) => {
            if (toolEvt.kind !== types_1.EditorEventType.ToolUpdated) {
                throw new Error('Invalid event type!');
            }
            // The button icon may depend on tool properties.
            if (toolEvt.tool === this.tool) {
                this.updateIcon();
                this.updateInputs();
            }
        });
    }
    getTitle() {
        return this.targetTool.description;
    }
    // Return the index of this tool's stroke factory in the list of
    // all stroke factories.
    //
    // Returns -1 if the stroke factory is not in the list of all stroke factories.
    getCurrentPenTypeIdx() {
        const currentFactory = this.tool.getStrokeFactory();
        for (let i = 0; i < this.penTypes.length; i++) {
            if (this.penTypes[i].factory === currentFactory) {
                return i;
            }
        }
        return -1;
    }
    getCurrentPenType() {
        for (const penType of this.penTypes) {
            if (penType.factory === this.tool.getStrokeFactory()) {
                return penType;
            }
        }
        return null;
    }
    createIconForRecord(record) {
        const style = {
            ...this.tool.getStyleValue().get(),
        };
        if (record?.factory) {
            style.factory = record.factory;
        }
        const strokeFactory = record?.factory;
        if (!strokeFactory ||
            strokeFactory === FreehandLineBuilder_1.makeFreehandLineBuilder ||
            strokeFactory === PressureSensitiveFreehandLineBuilder_1.makePressureSensitiveFreehandLineBuilder ||
            strokeFactory === PolylineBuilder_1.makePolylineBuilder) {
            return this.editor.icons.makePenIcon(style);
        }
        else {
            return this.editor.icons.makeIconFromFactory(style);
        }
    }
    createIcon() {
        return this.createIconForRecord(this.getCurrentPenType());
    }
    // Creates a widget that allows selecting different pen types
    createPenTypeSelector(helpOverlay) {
        const allChoices = this.penTypes.map((penType, index) => {
            return {
                id: index,
                makeIcon: () => this.createIconForRecord(penType),
                title: penType.name,
                isShapeBuilder: penType.isShapeBuilder ?? false,
            };
        });
        const penItems = allChoices.filter((choice) => !choice.isShapeBuilder);
        const penSelector = (0, makeGridSelector_1.default)(this.localizationTable.selectPenType, this.getCurrentPenTypeIdx(), penItems);
        const shapeItems = allChoices.filter((choice) => choice.isShapeBuilder);
        const shapeSelector = (0, makeGridSelector_1.default)(this.localizationTable.selectShape, this.getCurrentPenTypeIdx(), shapeItems);
        const onSelectorUpdate = (newPenTypeIndex) => {
            this.tool.setStrokeFactory(this.penTypes[newPenTypeIndex].factory);
        };
        penSelector.value.onUpdate(onSelectorUpdate);
        shapeSelector.value.onUpdate(onSelectorUpdate);
        helpOverlay?.registerTextHelpForElements([penSelector.getRootElement(), shapeSelector.getRootElement()], this.localizationTable.penDropdown__penTypeHelpText);
        return {
            setValue: (penTypeIndex) => {
                penSelector.value.set(penTypeIndex);
                shapeSelector.value.set(penTypeIndex);
            },
            updateIcons: () => {
                penSelector.updateIcons();
                shapeSelector.updateIcons();
            },
            addTo: (parent) => {
                if (penItems.length) {
                    penSelector.addTo(parent);
                }
                if (shapeItems.length) {
                    shapeSelector.addTo(parent);
                }
            },
        };
    }
    createStrokeCorrectionOptions(helpOverlay) {
        const container = document.createElement('div');
        container.classList.add('action-button-row', `${constants_1.toolbarCSSPrefix}-pen-tool-toggle-buttons`);
        const addToggleButton = (labelText, icon) => {
            const button = (0, createElement_1.default)('button', { type: 'button' });
            button.classList.add(`${constants_1.toolbarCSSPrefix}-toggle-button`);
            const iconElement = icon.cloneNode(true);
            iconElement.classList.add('icon');
            const label = document.createElement('span');
            label.innerText = labelText;
            button.replaceChildren(iconElement, label);
            button.setAttribute('role', 'switch');
            container.appendChild(button);
            let checked = false;
            let onChangeListener = (_checked) => { };
            const result = {
                setChecked(newChecked) {
                    checked = newChecked;
                    button.setAttribute('aria-checked', `${checked}`);
                    onChangeListener(checked);
                },
                setOnInputListener(listener) {
                    onChangeListener = listener;
                },
                addHelpText(text) {
                    helpOverlay?.registerTextHelpForElement(button, text);
                },
            };
            button.onclick = () => {
                result.setChecked(!checked);
            };
            return result;
        };
        const stabilizationOption = addToggleButton(this.localizationTable.inputStabilization, this.editor.icons.makeStrokeSmoothingIcon());
        stabilizationOption.setOnInputListener((enabled) => {
            this.tool.setHasStabilization(enabled);
        });
        const autocorrectOption = addToggleButton(this.localizationTable.strokeAutocorrect, this.editor.icons.makeShapeAutocorrectIcon());
        autocorrectOption.setOnInputListener((enabled) => {
            this.tool.setStrokeAutocorrectEnabled(enabled);
        });
        const pressureSensitivityOption = addToggleButton(this.localizationTable.pressureSensitivity, this.editor.icons.makePressureSensitivityIcon());
        pressureSensitivityOption.setOnInputListener((enabled) => {
            this.tool.setPressureSensitivityEnabled(enabled);
        });
        // Help text
        autocorrectOption.addHelpText(this.localizationTable.penDropdown__autocorrectHelpText);
        stabilizationOption.addHelpText(this.localizationTable.penDropdown__stabilizationHelpText);
        pressureSensitivityOption.addHelpText(this.localizationTable.penDropdown__pressureSensitivityHelpText);
        return {
            update: () => {
                stabilizationOption.setChecked(!!this.tool.getInputMapper());
                autocorrectOption.setChecked(this.tool.getStrokeAutocorrectionEnabled());
                pressureSensitivityOption.setChecked(this.tool.getPressureSensitivityEnabled());
            },
            addTo: (parent) => {
                parent.appendChild(container);
            },
        };
    }
    getHelpText() {
        return this.localizationTable.penDropdown__baseHelpText;
    }
    fillDropdown(dropdown, helpDisplay) {
        const container = document.createElement('div');
        container.classList.add(`${constants_1.toolbarCSSPrefix}spacedList`, `${constants_1.toolbarCSSPrefix}nonbutton-controls-main-list`);
        // Thickness: Value of the input is squared to allow for finer control/larger values.
        const { container: thicknessRow, setValue: setThickness } = (0, makeThicknessSlider_1.default)(this.editor, (thickness) => {
            this.tool.setThickness(thickness);
        });
        const colorRow = document.createElement('div');
        const colorLabel = document.createElement('label');
        const colorInputControl = (0, makeColorInput_1.default)(this.editor, (color) => {
            this.tool.setColor(color);
        });
        const { input: colorInput, container: colorInputContainer } = colorInputControl;
        colorInput.id = `${constants_1.toolbarCSSPrefix}colorInput${PenToolWidget.idCounter++}`;
        colorLabel.innerText = this.localizationTable.colorLabel;
        colorLabel.setAttribute('for', colorInput.id);
        colorRow.appendChild(colorLabel);
        colorRow.appendChild(colorInputContainer);
        // Autocorrect and stabilization options
        const toggleButtonRow = this.createStrokeCorrectionOptions(helpDisplay);
        const penTypeSelect = this.createPenTypeSelector(helpDisplay);
        // Add help text for color and thickness last, as these are likely to be
        // features users are least interested in.
        helpDisplay?.registerTextHelpForElement(colorRow, this.localizationTable.penDropdown__colorHelpText);
        if (helpDisplay) {
            colorInputControl.registerWithHelpTextDisplay(helpDisplay);
        }
        helpDisplay?.registerTextHelpForElement(thicknessRow, this.localizationTable.penDropdown__thicknessHelpText);
        this.updateInputs = () => {
            colorInputControl.setValue(this.tool.getColor());
            setThickness(this.tool.getThickness());
            penTypeSelect.updateIcons();
            // Update the selected stroke factory.
            penTypeSelect.setValue(this.getCurrentPenTypeIdx());
            toggleButtonRow.update();
        };
        this.updateInputs();
        container.replaceChildren(colorRow, thicknessRow);
        penTypeSelect.addTo(container);
        dropdown.replaceChildren(container);
        // Add the toggle button row *outside* of the main content (use different
        // spacing with respect to the sides of the container).
        toggleButtonRow.addTo(dropdown);
        return true;
    }
    onKeyPress(event) {
        if (!this.isSelected()) {
            return false;
        }
        for (let i = 0; i < keybindings_1.selectStrokeTypeKeyboardShortcutIds.length; i++) {
            const shortcut = keybindings_1.selectStrokeTypeKeyboardShortcutIds[i];
            if (this.editor.shortcuts.matchesShortcut(shortcut, event)) {
                const penTypeIdx = i;
                if (penTypeIdx < this.penTypes.length) {
                    this.tool.setStrokeFactory(this.penTypes[penTypeIdx].factory);
                    return true;
                }
            }
        }
        // Run any default actions registered by the parent class.
        if (super.onKeyPress(event)) {
            return true;
        }
        return false;
    }
    serializeState() {
        return {
            ...super.serializeState(),
            color: this.tool.getColor().toHexString(),
            thickness: this.tool.getThickness(),
            strokeFactoryId: this.getCurrentPenType()?.id,
            inputStabilization: !!this.tool.getInputMapper(),
            strokeAutocorrect: this.tool.getStrokeAutocorrectionEnabled(),
            pressureSensitivity: this.tool.getPressureSensitivityEnabled(),
        };
    }
    deserializeFrom(state) {
        super.deserializeFrom(state);
        const verifyPropertyType = (propertyName, expectedType) => {
            const actualType = typeof state[propertyName];
            if (actualType !== expectedType) {
                throw new Error(`Deserializing property ${propertyName}: Invalid type. Expected ${expectedType},` +
                    ` was ${actualType}.`);
            }
        };
        if (state.color) {
            verifyPropertyType('color', 'string');
            this.tool.setColor(math_1.Color4.fromHex(state.color));
        }
        if (state.thickness) {
            verifyPropertyType('thickness', 'number');
            this.tool.setThickness(state.thickness);
        }
        if (state.strokeFactoryId) {
            verifyPropertyType('strokeFactoryId', 'string');
            const factoryId = state.strokeFactoryId;
            for (const penType of this.penTypes) {
                if (factoryId === penType.id) {
                    this.tool.setStrokeFactory(penType.factory);
                    break;
                }
            }
        }
        if (state.inputStabilization !== undefined) {
            this.tool.setHasStabilization(!!state.inputStabilization);
        }
        if (state.strokeAutocorrect !== undefined) {
            this.tool.setStrokeAutocorrectEnabled(!!state.strokeAutocorrect);
        }
        if (state.pressureSensitivity !== undefined) {
            this.tool.setPressureSensitivityEnabled(!!state.pressureSensitivity);
        }
    }
}
// A counter variable that ensures different HTML elements are given unique names/ids.
PenToolWidget.idCounter = 0;
exports.default = PenToolWidget;
