"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const ImageComponent_1 = __importDefault(require("../../../components/ImageComponent"));
const Erase_1 = __importDefault(require("../../../commands/Erase"));
const EditorImage_1 = __importDefault(require("../../../image/EditorImage"));
const uniteCommands_1 = __importDefault(require("../../../commands/uniteCommands"));
const SelectionTool_1 = __importDefault(require("../../../tools/SelectionTool/SelectionTool"));
const math_1 = require("@js-draw/math");
const BaseWidget_1 = __importDefault(require("../BaseWidget"));
const types_1 = require("../../../types");
const constants_1 = require("../../constants");
const makeFileInput_1 = __importDefault(require("../components/makeFileInput"));
const ReactiveValue_1 = require("../../../util/ReactiveValue");
const bytesToSizeString_1 = __importDefault(require("../../../util/bytesToSizeString"));
const ImageWrapper_1 = require("./ImageWrapper");
const makeSnappedList_1 = __importDefault(require("../components/makeSnappedList"));
const fileToImages_1 = __importDefault(require("./fileToImages"));
const createButton_1 = __importDefault(require("../../../util/dom/createButton"));
/**
 * Provides a widget that allows inserting or modifying raster images.
 *
 * It's possible to customize the file picker used by this widget through {@link EditorSettings.image}.
 *
 * @example
 * ```ts,runnable
 * import { Editor, makeEdgeToolbar, InsertImageWidget } from 'js-draw';
 *
 * const editor = new Editor(document.body);
 * const toolbar = makeEdgeToolbar(editor);
 *
 * toolbar.addWidget(new InsertImageWidget(editor));
 * ```
 */
class InsertImageWidget extends BaseWidget_1.default {
    constructor(editor, localization) {
        localization ??= editor.localization;
        super(editor, 'insert-image-widget', localization);
        // Make the dropdown showable
        this.container.classList.add('dropdownShowable');
        editor.notifier.on(types_1.EditorEventType.SelectionUpdated, (event) => {
            if (event.kind === types_1.EditorEventType.SelectionUpdated && this.isDropdownVisible()) {
                this.updateInputs();
            }
        });
        this.images = ReactiveValue_1.MutableReactiveValue.fromInitialValue([]);
        this.images.onUpdateAndNow(() => {
            this.onImageDataUpdate();
        });
    }
    getTitle() {
        return this.localizationTable.image;
    }
    createIcon() {
        return this.editor.icons.makeInsertImageIcon();
    }
    setDropdownVisible(visible) {
        super.setDropdownVisible(visible);
        // Update the dropdown just before showing.
        if (this.isDropdownVisible()) {
            this.updateInputs();
        }
        else {
            // Allow any previously-selected files to be freed.
            this.selectedFiles?.set([]);
        }
    }
    handleClick() {
        this.setDropdownVisible(!this.isDropdownVisible());
    }
    fillDropdown(dropdown) {
        const container = document.createElement('div');
        container.classList.add('insert-image-widget-dropdown-content', `${constants_1.toolbarCSSPrefix}spacedList`, `${constants_1.toolbarCSSPrefix}nonbutton-controls-main-list`);
        const { container: chooseImageRow, selectedFiles } = (0, makeFileInput_1.default)(this.localizationTable.chooseFile, this.editor, {
            accepts: 'image/*',
            allowMultiSelect: true,
            customPickerAction: this.editor.getCurrentSettings().image?.showImagePicker,
        });
        const altTextRow = document.createElement('div');
        this.imagesPreview = (0, makeSnappedList_1.default)(this.images);
        this.statusView = document.createElement('div');
        const actionButtonRow = document.createElement('div');
        actionButtonRow.classList.add('action-button-row');
        this.statusView.classList.add('insert-image-image-status-view');
        this.submitButton = (0, createButton_1.default)();
        this.selectedFiles = selectedFiles;
        this.imageAltTextInput = document.createElement('input');
        // Label the alt text input
        const imageAltTextLabel = document.createElement('label');
        const altTextInputId = `insert-image-alt-text-input-${InsertImageWidget.nextInputId++}`;
        this.imageAltTextInput.setAttribute('id', altTextInputId);
        imageAltTextLabel.htmlFor = altTextInputId;
        imageAltTextLabel.innerText = this.localizationTable.inputAltText;
        this.imageAltTextInput.type = 'text';
        this.imageAltTextInput.placeholder = this.localizationTable.describeTheImage;
        this.statusView.setAttribute('aria-live', 'polite');
        this.submitButton.innerText = this.localizationTable.submit;
        this.imagesPreview.visibleItem.onUpdateAndNow(() => this.onImageDataUpdate());
        this.imageAltTextInput.oninput = () => {
            const currentImage = this.imagesPreview.visibleItem.get();
            if (currentImage) {
                currentImage.setAltText(this.imageAltTextInput.value);
                this.submitButton.style.display = '';
            }
        };
        this.selectedFiles.onUpdateAndNow(async (files) => {
            if (files.length === 0) {
                this.images.set([]);
                return;
            }
            const previews = (await Promise.all(files.map(async (imageFile) => {
                let renderableImages;
                try {
                    renderableImages = await (0, fileToImages_1.default)(imageFile);
                }
                catch (error) {
                    console.error('Image load error', error);
                    const errorMessage = this.localizationTable.imageLoadError(error);
                    this.statusView.innerText = errorMessage;
                    return [];
                }
                return renderableImages.map((image) => {
                    const { wrapper, preview } = ImageWrapper_1.ImageWrapper.fromRenderable(image, () => this.onImageDataUpdate());
                    return {
                        data: wrapper,
                        element: preview,
                    };
                });
            }))).flat();
            this.images.set(previews);
        });
        altTextRow.replaceChildren(imageAltTextLabel, this.imageAltTextInput);
        actionButtonRow.replaceChildren(this.submitButton);
        container.replaceChildren(chooseImageRow, altTextRow, this.imagesPreview.container, this.statusView, actionButtonRow);
        dropdown.replaceChildren(container);
        return true;
    }
    onImageDataUpdate() {
        if (!this.imagesPreview)
            return;
        const currentImage = this.imagesPreview.visibleItem.get();
        const base64Data = currentImage?.getBase64Url();
        this.imageAltTextInput.value = currentImage?.getAltText() ?? '';
        if (base64Data) {
            this.submitButton.disabled = false;
            this.submitButton.style.display = '';
            this.updateImageSizeDisplay();
        }
        else {
            this.submitButton.disabled = true;
            this.submitButton.style.display = 'none';
            this.statusView.innerText = '';
            this.submitButton.disabled = true;
        }
        if (this.images.get().length <= 1) {
            this.submitButton.innerText = this.localizationTable.submit;
        }
        else {
            this.submitButton.innerText = this.localizationTable.addAll;
        }
    }
    hideDialog() {
        this.setDropdownVisible(false);
    }
    updateImageSizeDisplay() {
        const currentImage = this.imagesPreview.visibleItem.get();
        const imageData = currentImage?.getBase64Url() ?? '';
        const { size, units } = (0, bytesToSizeString_1.default)(imageData.length);
        const sizeText = document.createElement('span');
        sizeText.innerText = this.localizationTable.imageSize(Math.round(size), units);
        // Add a button to allow decreasing the size of large images.
        const decreaseSizeButton = (0, createButton_1.default)({
            text: this.localizationTable.decreaseImageSize,
            onClick: () => {
                currentImage?.decreaseSize();
            },
        });
        const resetSizeButton = (0, createButton_1.default)({
            text: this.localizationTable.resetImage,
            onClick: () => {
                currentImage?.reset();
            },
        });
        this.statusView.replaceChildren(sizeText);
        if (currentImage?.isLarge()) {
            this.statusView.appendChild(decreaseSizeButton);
        }
        else if (currentImage?.isChanged()) {
            this.statusView.appendChild(resetSizeButton);
        }
        else {
            const hasLargeOrChangedImages = this.images
                .get()
                .some((image) => image.data?.isChanged() || image.data?.isLarge());
            if (hasLargeOrChangedImages) {
                // Still show the button -- prevents the layout from readjusting while
                // scrolling through the image list
                decreaseSizeButton.disabled = true;
                this.statusView.appendChild(decreaseSizeButton);
            }
        }
    }
    updateInputs() {
        const resetInputs = () => {
            this.selectedFiles?.set([]);
            this.imageAltTextInput.value = '';
            this.submitButton.disabled = true;
            this.statusView.innerText = '';
            this.submitButton.style.display = '';
        };
        resetInputs();
        const selectionTools = this.editor.toolController.getMatchingTools(SelectionTool_1.default);
        const selectedObjects = selectionTools.map((tool) => tool.getSelectedObjects()).flat();
        // Check: Is there a selected image that can be edited?
        let editingImage = null;
        if (selectedObjects.length === 1 && selectedObjects[0] instanceof ImageComponent_1.default) {
            editingImage = selectedObjects[0];
            const image = new Image();
            const imageWrapper = ImageWrapper_1.ImageWrapper.fromSrcAndPreview(editingImage.getURL(), image, () => this.onImageDataUpdate());
            imageWrapper.setAltText(editingImage.getAltText() ?? '');
            this.images.set([{ data: imageWrapper, element: image }]);
        }
        else if (selectedObjects.length > 0) {
            // If not, clear the selection.
            selectionTools.forEach((tool) => tool.clearSelection());
        }
        // Show the submit button only when there is data to submit.
        this.submitButton.style.display = 'none';
        this.submitButton.onclick = async () => {
            const newComponents = [];
            let transform = math_1.Mat33.identity;
            let fullBBox = null;
            for (const { data: imageWrapper } of this.images.get()) {
                if (!imageWrapper) {
                    continue;
                }
                const image = new Image();
                image.src = imageWrapper.getBase64Url();
                const altText = imageWrapper.getAltText();
                if (altText) {
                    image.setAttribute('alt', altText);
                }
                let component;
                try {
                    component = await ImageComponent_1.default.fromImage(image, transform);
                }
                catch (error) {
                    console.error('Error loading image', error);
                    this.statusView.innerText = this.localizationTable.imageLoadError(error);
                    return;
                }
                const componentBBox = component.getBBox();
                if (componentBBox.area === 0) {
                    this.statusView.innerText = this.localizationTable.errorImageHasZeroSize;
                    return;
                }
                newComponents.push(component);
                fullBBox ??= componentBBox;
                fullBBox.union(componentBBox);
                // Update the transform for the next item.
                const shift = math_1.Vec2.of(0, componentBBox.height);
                transform = transform.rightMul(math_1.Mat33.translation(shift));
            }
            if (newComponents.length) {
                if (!fullBBox) {
                    throw new Error('Logic error: Full bounding box must be calculated when components are to be added.');
                }
                this.hideDialog();
                if (editingImage) {
                    const eraseCommand = new Erase_1.default([editingImage]);
                    // Try to preserve the original width
                    const originalTransform = editingImage.getTransformation();
                    // || 1: Prevent division by zero
                    const originalWidth = editingImage.getBBox().width || 1;
                    const newWidth = fullBBox.transformedBoundingBox(originalTransform).width || 1;
                    const widthAdjustTransform = math_1.Mat33.scaling2D(originalWidth / newWidth);
                    const commands = [];
                    for (const component of newComponents) {
                        commands.push(EditorImage_1.default.addComponent(component), component.transformBy(originalTransform.rightMul(widthAdjustTransform)), component.setZIndex(editingImage.getZIndex()));
                    }
                    this.editor.dispatch((0, uniteCommands_1.default)([...commands, eraseCommand]));
                    selectionTools[0]?.setSelection(newComponents);
                }
                else {
                    await this.editor.addAndCenterComponents(newComponents);
                }
            }
        };
    }
}
InsertImageWidget.nextInputId = 0;
exports.default = InsertImageWidget;
