"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const Eraser_1 = require("../../tools/Eraser");
const types_1 = require("../../types");
const constants_1 = require("../constants");
const BaseToolWidget_1 = __importDefault(require("./BaseToolWidget"));
const makeThicknessSlider_1 = __importDefault(require("./components/makeThicknessSlider"));
class EraserToolWidget extends BaseToolWidget_1.default {
    constructor(editor, tool, localizationTable) {
        super(editor, tool, 'eraser-tool-widget', localizationTable);
        this.tool = tool;
        this.updateInputs = () => { };
        this.editor.notifier.on(types_1.EditorEventType.ToolUpdated, (toolEvt) => {
            if (toolEvt.kind === types_1.EditorEventType.ToolUpdated && toolEvt.tool === this.tool) {
                this.updateInputs();
                this.updateIcon();
            }
        });
    }
    getHelpText() {
        return this.localizationTable.eraserDropdown__baseHelpText;
    }
    getTitle() {
        return this.localizationTable.eraser;
    }
    makeIconForType(mode) {
        return this.editor.icons.makeEraserIcon(this.tool.getThickness(), mode);
    }
    createIcon() {
        return this.makeIconForType(this.tool.getModeValue().get());
    }
    makeEraserTypeSelector(helpDisplay) {
        const container = document.createElement('div');
        const labelElement = document.createElement('label');
        const checkboxElement = document.createElement('input');
        checkboxElement.id = `${constants_1.toolbarCSSPrefix}eraserToolWidget-${EraserToolWidget.idCounter++}`;
        labelElement.htmlFor = checkboxElement.id;
        labelElement.innerText = this.localizationTable.fullStrokeEraser;
        checkboxElement.type = 'checkbox';
        checkboxElement.oninput = () => {
            this.tool
                .getModeValue()
                .set(checkboxElement.checked ? Eraser_1.EraserMode.FullStroke : Eraser_1.EraserMode.PartialStroke);
        };
        const updateValue = () => {
            checkboxElement.checked = this.tool.getModeValue().get() === Eraser_1.EraserMode.FullStroke;
        };
        container.replaceChildren(labelElement, checkboxElement);
        helpDisplay?.registerTextHelpForElement(container, this.localizationTable.eraserDropdown__fullStrokeEraserHelpText);
        return {
            addTo: (parent) => {
                parent.appendChild(container);
            },
            updateValue,
        };
    }
    fillDropdown(dropdown, helpDisplay) {
        const container = document.createElement('div');
        container.classList.add(`${constants_1.toolbarCSSPrefix}spacedList`, `${constants_1.toolbarCSSPrefix}nonbutton-controls-main-list`);
        const thicknessSlider = (0, makeThicknessSlider_1.default)(this.editor, (thickness) => {
            this.tool.setThickness(thickness);
        });
        thicknessSlider.setBounds(10, 55);
        helpDisplay?.registerTextHelpForElement(thicknessSlider.container, this.localizationTable.eraserDropdown__thicknessHelpText);
        const modeSelector = this.makeEraserTypeSelector(helpDisplay);
        this.updateInputs = () => {
            thicknessSlider.setValue(this.tool.getThickness());
            modeSelector.updateValue();
        };
        this.updateInputs();
        container.replaceChildren(thicknessSlider.container);
        modeSelector.addTo(container);
        dropdown.replaceChildren(container);
        return true;
    }
    serializeState() {
        return {
            ...super.serializeState(),
            thickness: this.tool.getThickness(),
            mode: this.tool.getModeValue().get(),
        };
    }
    deserializeFrom(state) {
        super.deserializeFrom(state);
        if (state.thickness) {
            const parsedThickness = parseFloat(state.thickness);
            if (typeof parsedThickness !== 'number' || !isFinite(parsedThickness)) {
                throw new Error(`Deserializing property ${parsedThickness} is not a number or is not finite.`);
            }
            this.tool.setThickness(parsedThickness);
        }
        if (state.mode) {
            const mode = state.mode;
            if (Object.values(Eraser_1.EraserMode).includes(mode)) {
                this.tool.getModeValue().set(mode);
            }
        }
    }
}
EraserToolWidget.idCounter = 0;
exports.default = EraserToolWidget;
