import { Color4, Mat33, Point2, Vec2, Rect2 } from '@js-draw/math';
import { LoadSaveDataTable } from '../../components/AbstractComponent';
import Viewport from '../../Viewport';
import RenderingStyle, { StrokeStyle } from '../RenderingStyle';
import TextRenderingStyle from '../TextRenderingStyle';
import RenderablePathSpec from '../RenderablePathSpec';
export interface RenderableImage {
    transform: Mat33;
    image: HTMLImageElement | HTMLCanvasElement;
    base64Url: string;
    label?: string;
}
/**
 * An interface that allows renderers to provide accelerated inking.
 *
 * This API is intended to reduce latency by rendering a draft preview of the
 * last, trailing part of a stroke.
 *
 * This may be implemented with, for example, [the web `Ink` API](https://developer.mozilla.org/en-US/docs/Web/API/Ink),
 * or do nothing at all, depending on the renderer and platform.
 */
export interface DraftInkPresenter {
    /** Enables the ink presenter, if available on the current platform. */
    setEnabled(pointerId: number, enabled: boolean): void;
    /** Sets the color and width of the stroke trail. */
    updateStyle(style: StrokeStyle): void;
}
/**
 * Abstract base class for renderers.
 *
 * @see {@link EditorImage.render}
 */
export default abstract class AbstractRenderer {
    private viewport;
    private selfTransform;
    private transformStack;
    protected constructor(viewport: Viewport);
    /**
     * this.canvasToScreen, etc. should be used instead of the corresponding
     * methods on `Viewport`, because the viewport may not accurately reflect
     * what is rendered.
     */
    protected getViewport(): Viewport;
    abstract displaySize(): Vec2;
    abstract clear(): void;
    protected abstract beginPath(startPoint: Point2): void;
    protected abstract endPath(style: RenderingStyle): void;
    protected abstract lineTo(point: Point2): void;
    protected abstract moveTo(point: Point2): void;
    protected abstract traceCubicBezierCurve(p1: Point2, p2: Point2, p3: Point2): void;
    protected abstract traceQuadraticBezierCurve(controlPoint: Point2, endPoint: Point2): void;
    abstract drawText(text: string, transform: Mat33, style: TextRenderingStyle): void;
    abstract drawImage(image: RenderableImage): void;
    abstract isTooSmallToRender(rect: Rect2): boolean;
    setDraftMode(_draftMode: boolean): void;
    /**
     * Returns an API that can be used to accelerate inking.
     */
    getDraftInkPresenter(): DraftInkPresenter;
    protected objectLevel: number;
    private currentPaths;
    private flushPath;
    /**
     * Draws a styled path. If within an object started by {@link startObject},
     * the resultant path may not be visible until {@link endObject} is called.
     */
    drawPath(path: RenderablePathSpec): void;
    drawRect(rect: Rect2, lineWidth: number, lineFill: RenderingStyle): void;
    /** Draws a filled rectangle. */
    fillRect(rect: Rect2, fill: Color4): void;
    /**
     * This should be called whenever a new object is being drawn.
     *
     * @param _boundingBox The bounding box of the object to be drawn.
     * @param _clip Whether content outside `_boundingBox` should be drawn. Renderers
     *              that override this method are not required to support `_clip`.
     */
    startObject(_boundingBox: Rect2, _clip?: boolean): void;
    /**
     * Notes the end of an object.
     * @param _loaderData - a map from strings to JSON-ifyable objects
     * and contains properties attached to the object by whatever loader loaded the image. This
     * is used to preserve attributes not supported by js-draw when loading/saving an image.
     * Renderers may ignore this.
     *
     * @param _objectTags - a list of labels (e.g. `className`s) to be attached to the object.
     * Renderers may ignore this.
     */
    endObject(_loaderData?: LoadSaveDataTable, _objectTags?: string[]): void;
    protected getNestingLevel(): number;
    abstract drawPoints(...points: Point2[]): void;
    canRenderFromWithoutDataLoss(_other: AbstractRenderer): boolean;
    renderFromOtherOfSameType(_renderTo: Mat33, other: AbstractRenderer): void;
    setTransform(transform: Mat33 | null): void;
    pushTransform(transform: Mat33): void;
    popTransform(): void;
    getCanvasToScreenTransform(): Mat33;
    canvasToScreen(vec: Vec2): Vec2;
    getSizeOfCanvasPixelOnScreen(): number;
    private visibleRectOverride;
    /**
     * @internal
     */
    overrideVisibleRect(rect: Rect2 | null): void;
    getVisibleRect(): Rect2;
}
